/*
 * Copyright 2014 Attila Szegedi, Daniel Dekany, Jonathan Revusky
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package freemarker.template.utility;

import java.math.BigDecimal;
import java.math.BigInteger;

/**
 * Number- and math-related utilities.
 * 
 * @since 2.3.20
 */
public class NumberUtil {

    private NumberUtil() { }
    
    public static boolean isInfinite(Number num) {
        if (num instanceof Double) {
            return ((Double) num).isInfinite();
        } else if (num instanceof Float) {
            return ((Float) num).isInfinite();
        } else if (isNonFPNumberOfSupportedClass(num)) {
            return false;
        } else {
            throw new UnsupportedNumberClassException(num.getClass());
        }           
    }

    public static boolean isNaN(Number num) {
        if (num instanceof Double) {
            return ((Double) num).isNaN();
        } else if (num instanceof Float) {
            return ((Float) num).isNaN();
        } else if (isNonFPNumberOfSupportedClass(num)) {
            return false;
        } else {
            throw new UnsupportedNumberClassException(num.getClass());
        }           
    }

    /**
     * @return -1 for negative, 0 for zero, 1 for positive.
     * @throws ArithmeticException if the number is NaN
     */
    public static int getSignum(Number num) throws ArithmeticException {
        if (num instanceof Integer) {
            int n = ((Integer) num).intValue();
            return n > 0 ? 1 : (n == 0 ? 0 : -1);
        } else if (num instanceof BigDecimal) {
            BigDecimal n = (BigDecimal) num;
            return n.signum();
        } else if (num instanceof Double) {
            double n = ((Double) num).doubleValue();
            if (n > 0) return 1;
            else if (n == 0) return 0;
            else if (n < 0) return -1;
            else throw new ArithmeticException("The signum of " + n + " is not defined.");  // NaN
        } else if (num instanceof Float) {
            float n = ((Float) num).floatValue();
            if (n > 0) return 1;
            else if (n == 0) return 0;
            else if (n < 0) return -1;
            else throw new ArithmeticException("The signum of " + n + " is not defined.");  // NaN
        } else if (num instanceof Long) {
            long n = ((Long) num).longValue();
            return n > 0 ? 1 : (n == 0 ? 0 : -1);
        } else if (num instanceof Short) {
            short n = ((Short) num).shortValue();
            return n > 0 ? 1 : (n == 0 ? 0 : -1);
        } else if (num instanceof Byte) {
            byte n = ((Byte) num).byteValue();
            return n > 0 ? 1 : (n == 0 ? 0 : -1);
        } else if (num instanceof BigInteger) {
            BigInteger n = (BigInteger) num;
            return n.signum();
        } else {
            throw new UnsupportedNumberClassException(num.getClass());
        }
    }
    
    /**
     * Tells if a {@link BigDecimal} stores a whole number. For example, it returns {@code true} for {@code 1.0000},
     * but {@code false} for {@code 1.0001}.
     * 
     * @since 2.3.21
     */
    static public boolean isIntegerBigDecimal(BigDecimal bd) {
            return bd.scale() <= 0  // A fast check that whole numbers usually (not always) match
                   || bd.setScale(0, BigDecimal.ROUND_DOWN).compareTo(bd) == 0;  // This is rather slow
            // Note that `bd.signum() == 0 || bd.stripTrailingZeros().scale() <= 0` was also tried for the last
            // condition, but stripTrailingZeros was slower than setScale + compareTo.
    }
    
    private static boolean isNonFPNumberOfSupportedClass(Number num) {
        return num instanceof Integer || num instanceof BigDecimal || num instanceof Long
                || num instanceof Short || num instanceof Byte || num instanceof BigInteger;
    }
    
}
