/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.fryske_akademy.ejb;

/*-
 * #%L
 * ejbCrudApi
 * %%
 * Copyright (C) 2018 Fryske Akademy
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.security.DeclareRoles;
import javax.annotation.security.RolesAllowed;
import javax.ejb.EJB;
import javax.ejb.TransactionAttribute;
import javax.ejb.TransactionAttributeType;
import javax.persistence.EntityManager;
import javax.persistence.NoResultException;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.CriteriaQuery;
import org.fryske_akademy.jpa.EntityInterface;
import org.fryske_akademy.jpa.RevInfo;
import org.fryske_akademy.jpa.RevisionInfo;
import org.hibernate.envers.AuditReaderFactory;
import org.hibernate.envers.Audited;
import org.hibernate.envers.RevisionType;
import org.hibernate.envers.query.AuditEntity;
import org.hibernate.envers.query.AuditQuery;
import org.hibernate.envers.query.criteria.internal.IdentifierEqAuditExpression;

/**
 * <p>This base class enables you to create minimal crud service beans. All you have to do is
 * override, implement getEntityManager and add annotation @Stateless, @Local(CrudReadService.class, CrudWriteService.class).</p>
 * <p>Inject the crud interfaces using @Inject in your frontend (problems with inhertitance and @EJB).</p>
 * <p>This base class declares appropriate transactions and an EDITOR role to protect write operations.</p>
 * <p>An injected {@link JpqlBuilder} is used for building where and order by clauses in dynamic queries, i.e. used in 
 * {@link #findPaged(int, int, java.util.Map, java.util.List, java.lang.Class) } and</p>
 * {@link #find(java.util.Map, java.util.List, java.lang.Integer, java.lang.Class) }. The JpqlBuilder is also used for setting parameter values.
 * @author eduard
 */
@DeclareRoles("editor")
@TransactionAttribute(TransactionAttributeType.NOT_SUPPORTED)
public abstract class AbstractCrudService  implements CrudReadService, CrudWriteService {
    
    @EJB
    private JpqlBuilder jpqlBuilder;
    
    private static final Logger LOGGER = Logger.getLogger(AbstractCrudService.class.getName());
    
    protected abstract EntityManager getEntityManager();
    
    @Override
    public <T extends EntityInterface> T find(Serializable id, Class<T> type) {
        return getEntityManager().find(type, id);
    }
    
    @Override
    public <T extends EntityInterface> List<T> findAll(Class<T> type) {
        CriteriaQuery cq = getEntityManager().getCriteriaBuilder().createQuery();
        cq.select(cq.from(type));
        return getEntityManager().createQuery(cq).getResultList();
    }
    
    @Override
    public <T extends EntityInterface> List<T> findPaged(int first, int pageSize, Map<String, SORTORDER> sort, List<Param> params, Class<T> type) {
        String jpql = "from " + type.getSimpleName() + " e" + jpqlBuilder.whereClause(params) + jpqlBuilder.orderClause(sort);
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.fine(jpql);
        }
        TypedQuery<T> q = getEntityManager().createQuery(jpql, type);
        jpqlBuilder.setWhereParams(q, params);
        q.setFirstResult(first);
        q.setMaxResults(pageSize);
        return (List<T>) q.getResultList();
    }
    
    @Override
    public <T extends EntityInterface> List<T> find(Map<String, SORTORDER> sort, List<Param> params, Integer max, Class<T> type) {
        String jpql = "from " + type.getSimpleName() + " e" + jpqlBuilder.whereClause(params) + jpqlBuilder.orderClause(sort);
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.fine(jpql);
        }
        TypedQuery<T> q = getEntityManager().createQuery(jpql, type);
        jpqlBuilder.setWhereParams(q, params);
        if (max!=null) q.setMaxResults(max);
        return (List<T>) q.getResultList();
    }
    
    @Override
    public <T extends EntityInterface> List<T> find(String namedQuery, List<Param> params, Integer max, Class<T> type) {
        TypedQuery<T> q = getEntityManager().createNamedQuery(namedQuery, type);
        jpqlBuilder.setWhereParams(q, params);
        if (max!=null) q.setMaxResults(max);
        return q.getResultList();
    }

    @Override
    public <T extends EntityInterface> List<T> autoComplete(String namedQuery, String paramName, String paramValue, Integer max, Class<T> type) {
        return find(namedQuery, new Param.Builder().add(paramName, paramValue).build(), (max==null)?MAX_RESULT_AUTOCOMPLETE:null, type);
    }
    public static final int MAX_RESULT_AUTOCOMPLETE = 30;

    @Override
    public <T extends EntityInterface> T findOne(String namedQuery, List<Param> params, Class<T> type) {
        TypedQuery<T> q = getEntityManager().createNamedQuery(namedQuery, type);
        jpqlBuilder.setWhereParams(q, params);
        try {
            return q.getSingleResult();
        } catch (NoResultException e) {
            return null;
        }
    }
    
    @Override
    public <T extends EntityInterface> T findOne(String namedQuery, String paramName, Object paramValue, Class<T> type) {
        return findOne(namedQuery,new Param.Builder().add(paramName, paramValue).build(),  type);
    }
    @Override
    public int count(List<Param> params, Class type) {
        String jpql = "select count(*) from " + type.getSimpleName() + " e" + jpqlBuilder.whereClause(params);
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.fine(jpql);
        }
        TypedQuery<Long> q = getEntityManager().createQuery(jpql, Long.class);
        jpqlBuilder.setWhereParams(q, params);
        return q.getSingleResult().intValue();
    }
    
    @Override
    public <T extends EntityInterface> T getRevision(Number n, Class<T> type) {
        if (!type.isAnnotationPresent(Audited.class)) {
            if (LOGGER.isLoggable(Level.FINE)) {
                LOGGER.fine("No @Audited annotation on " + type.getName());
            }
            return null;
        }
        return (T) AuditReaderFactory.get(getEntityManager()).createQuery().forEntitiesAtRevision(type, n).getSingleResult();
    }
    
    @Override
    public <T extends EntityInterface> List<RevInfo<T>> getRevisionInfo(T entity, Integer max, Class<T> type) {
        List<RevInfo<T>> result = new ArrayList<>(5);
        if (!type.isAnnotationPresent(Audited.class)) {
            if (LOGGER.isLoggable(Level.FINE)) {
                LOGGER.fine("No @Audited annotation on " + type.getName());
            }
            return result;
        }
        if (entity==null) {
            return result;
        }
        AuditQuery query = AuditReaderFactory.get(getEntityManager()).createQuery().forRevisionsOfEntity(type, false, true).addOrder(AuditEntity.revisionNumber().desc());
        query.add(new IdentifierEqAuditExpression(null, entity.getId(), true)).setMaxResults((max == null) ? 5 : max);
        List<Object[]> resultList = query.getResultList();
        if (resultList.isEmpty()) {
            return result;
        }
        for (Object[] object : resultList) {
            result.add(new RevInfo<T>((T) object[0], (RevisionType) object[2], (RevisionInfo) object[1]));
        }
        return result;
    }
    public static final String EDITOR = "editor";
    
    @Override
    @RolesAllowed(value = {EDITOR})
    @TransactionAttribute(TransactionAttributeType.REQUIRED)
    public <T extends EntityInterface> T createOrUpdate(T t) {
        if (t.isTransient()) {
            getEntityManager().persist(t);
            return t;
        } else {
            return getEntityManager().merge(t);
        }
    }
    
    @Override
    @RolesAllowed(value = {EDITOR})
    @TransactionAttribute(TransactionAttributeType.REQUIRED)
    public void delete(EntityInterface t) {
        getEntityManager().remove(getEntityManager().merge(t));
    }
    
    @Override
    public <T extends EntityInterface> List<Number> getRevisionNumbers(Object id, Class<T> type) {
        if (!type.isAnnotationPresent(Audited.class)) {
            return Collections.emptyList();
        }
        return AuditReaderFactory.get(getEntityManager()).getRevisions(type, id);
    }

}
