/*
 * Copyright 2018 Fryske Akademy.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.fryske_akademy.ejb;

/*-
 * #%L
 * ejbCrudApi
 * %%
 * Copyright (C) 2018 Fryske Akademy
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.ejb.Local;
import javax.ejb.Stateless;
import javax.persistence.TypedQuery;
import org.fryske_akademy.ejb.CrudReadService.SORTORDER;

/**
 * contains function for building dynamic jpql where and order by clause based on {@link Param}
 * and functions for setting parameters in a TypedQuery, also based on {@link Param}. Supports all
 * comparison expressions except between when building jpql.
 * @author eduard
 */
@Stateless
@Local(JpqlBuilder.class)
public class JpqlBuilderImpl implements JpqlBuilder {
    
    private static final Logger LOGGER = Logger.getLogger(JpqlBuilder.class.getName());

    /**
     * tabel alias used in building where and order by clause.
     */
    public static final String TABLE_ALIAS = "e";

    /**
     *
     * builds an order by clause, uses {@link #TABLE_ALIAS} as alias for the
     * table
     *
     * @param sort
     * @return
     */
    @Override
    public String orderClause(Map<String, SORTORDER> sort) {
        if (sort != null && !sort.isEmpty()) {
            String srt = "";
            boolean first = true;
            for (Map.Entry<String, SORTORDER> entry : sort.entrySet()) {
                String sortField = entry.getKey();
                if (sortField == null) {
                    continue;
                }
                srt += (first) ? " order by" : ",";
                srt += " e." + sortField + " " + entry.getValue();
                first = false;
            }
            return srt;
        }
        return "";
    }

    /**
     * builds a where clause, calls {@link #whereCondition(org.fryske_akademy.ejb.Param) 
     * }
     * for every entry, uses {@link #TABLE_ALIAS} as alias for the table
     *
     * @param params
     * @return
     */
    @Override
    public String whereClause(List<Param> params) {
        if (params != null && !params.isEmpty()) {
            String where = "";
            boolean first = true;
            for (Param param : params) {
                where += ((first) ? " where" : param.getAndOr()) + whereCondition(param);
                first = false;
            }
            return where;
        }
        return "";
    }

    /**
     * builds a where condition, prepares ql parameters later filled in
     * {@link #setWhereParams(javax.persistence.TypedQuery, java.util.List) }.
     * Supports all comparison operators in {@link Param#getOperator() } except between, for null and empty comparison {@link Param#getNot() }
     * and {@link Param#getParamValue() } are ignored.
     *
     * @param param
     * @return
     */
    @Override
    public String whereCondition(Param param) {
        if (param.getOperator().toLowerCase().contains("member of")) {
            return " :" + param.getParamKey() + param.getNot() + param.getOperator() + " e." + param.getPropertyPath();
        } else if (param.getOperator().toLowerCase().contains("in")) {
            return " e." + param.getPropertyPath() + param.getNot() + param.getOperator() + "( :" + param.getParamKey() + " )";
        } else if (param.getOperator().toLowerCase().contains("null") || param.getOperator().toLowerCase().contains("empty")) {
            return " e." + param.getPropertyPath() + param.getOperator();
        } else {
            return param.getNot() + " e." + param.getPropertyPath() + param.getOperator() + ":" + param.getParamKey();
        }
    }

    /**
     * Calls {@link #setParam(javax.persistence.TypedQuery, org.fryske_akademy.ejb.Param) 
     * } for each filter.
     *
     * @param q
     * @param params
     */
    @Override
    public void setWhereParams(TypedQuery q, List<Param> params) {
        if (params != null && !params.isEmpty()) {
            for (Param param : params) {
                setParam(q, param);
            }
        }
    }

    /**
     * Fills parameters prepared in {@link #setWhereParams(javax.persistence.TypedQuery, java.util.List)
     * }, if the type of the field in the query is a Short or an Integer and the
     * paramValue is a String, it is converted accordingly. Via Param you
 have full control over the Object that is used as paramValue.
     *
     * @param q
     * @param param
     */
    @Override
    public void setParam(TypedQuery q, Param param) {
        if (param.isSkipSetValue()) {
            if (LOGGER.isLoggable(Level.FINE)) {
                LOGGER.fine("skipping " + param);
            }
            return;
        }
        if (Short.class.isAssignableFrom(q.getParameter(param.getParamKey()).getParameterType()) && param.getParamValue() instanceof String) {
            q.setParameter(param.getParamKey(), Short.valueOf(String.valueOf(param.getParamValue())));
        } else if (Integer.class.isAssignableFrom(q.getParameter(param.getParamKey()).getParameterType()) && param.getParamValue() instanceof String) {
            q.setParameter(param.getParamKey(), Integer.valueOf(String.valueOf(param.getParamValue())));
        } else {
            q.setParameter(param.getParamKey(), param.getParamValue());
        }
    }

}
