/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.fryske_akademy.ejb;

/*-
 * #%L
 * ejbCrudApi
 * %%
 * Copyright (C) 2018 Fryske Akademy
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */
import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.security.DeclareRoles;
import javax.annotation.security.PermitAll;
import javax.annotation.security.RolesAllowed;
import javax.ejb.TransactionAttribute;
import javax.persistence.EntityManager;
import javax.persistence.NoResultException;
import javax.persistence.Query;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.CriteriaQuery;
import javax.transaction.Transactional;

/**
 * <p>
 * This base class enables you to create minimal crud service beans. All you
 * have to do is override, implement getEntityManager and annotate with
 *
 * @Stateless, @Local(CrudReadService.class, CrudWriteService.class).</p>
 * <p>
 * Inject the crud interfaces using @Inject where you need them.</p>
 * <p>
 * This base class declares {@link Transactional} and {@link #EDITORROLE} to
 * protect write operations. When overriding don't forget transaction and role
 * annotations are not inherited. If you don't need security, override and use
 * {@link PermitAll}.</p>
 * <p>
 * A {@link JpqlBuilderImpl} is used for building where and order by clauses in
 * dynamic queries, i.e. used in  {@link #findDynamic(java.lang.Integer, java.lang.Integer, java.util.Map, java.util.List, java.lang.Class)
 * }.</p>
 * The JpqlBuilder is also used for setting parameter values.
 *
 * @author eduard
 */
@DeclareRoles("editor")
public abstract class AbstractCrudService implements CrudReadService, CrudWriteService {

    private static final JpqlBuilder jpqlBuilder = new JpqlBuilderImpl();

    private static final Logger LOGGER = Logger.getLogger(AbstractCrudService.class.getName());

    protected abstract EntityManager getEntityManager();

    @Override
    public <T extends Serializable> T find(Serializable id, Class<T> type) {
        return getEntityManager().find(type, id);
    }

    @Override
    public <T extends Serializable> List<T> findAll(Class<T> type) {
        CriteriaQuery cq = getEntityManager().getCriteriaBuilder().createQuery();
        cq.select(cq.from(type));
        return getEntityManager().createQuery(cq).getResultList();
    }

    /**
     *
     * @param <T>
     * @param first defaults to 0
     * @param pageSize defaults to {@link #DEFAULT_PAGE_SIZE}, negative value
     * means no limit
     * @param sort
     * @param params
     * @param type
     * @return
     */
    @Override
    public <T extends Serializable> List<T> findDynamic(Integer first, Integer pageSize, Map<String, SORTORDER> sort, List<Param> params, Class<T> type) {
        String jpql = "from " + type.getSimpleName() + " e" + jpqlBuilder.whereClause(params) + jpqlBuilder.orderClause(sort);
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.fine(jpql);
        }
        TypedQuery<T> q = getEntityManager().createQuery(jpql, type);
        jpqlBuilder.setWhereParams(q, params);
        q.setFirstResult(first == null ? 0 : first);
        if (!(pageSize != null && pageSize < 0)) {
            q.setMaxResults(pageSize == null ? DEFAULT_PAGE_SIZE : pageSize);
        }
        return (List<T>) q.getResultList();
    }

    @Override
    public int countDynamic(List<Param> params, Class type) {
        String jpql = "select count(*) from " + type.getSimpleName() + " e" + jpqlBuilder.whereClause(params);
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.fine(jpql);
        }
        return doCount(getEntityManager().createQuery(jpql, Long.class), params);
    }

    /**
     *
     * @param <T>
     * @param namedQuery
     * @param params
     * @param first defaults to 0
     * @param max defaults to {@link #DEFAULT_PAGE_SIZE}, negative value means
     * no limit
     * @param type
     * @return
     */
    @Override
    public <T extends Serializable> List<T> find(String namedQuery, List<Param> params, Integer first, Integer max, Class<T> type) {
        return doFind(getEntityManager().createNamedQuery(namedQuery, type), params, first, max);
    }

    /**
     *
     * @param <T>
     * @param namedNativeQuery
     * @param params
     * @param first
     * @param max defaults to {@link #DEFAULT_PAGE_SIZE}, negative value means
     * no limit
     * @param type
     * @return
     */
    @Override
    public <T extends Serializable> List<T> findNative(String namedNativeQuery, List<Param> params, Integer first, Integer max, Class<T> type) {
        return doFind(getEntityManager().createNamedQuery(namedNativeQuery), params, first, max);
    }

    @Override
    public int count(String namedQuery, List<Param> params) {
        return doCount(getEntityManager().createNamedQuery(namedQuery), params);
    }

    /**
     * default page size (max results), used when no maximum is given.
     */
    public static final int DEFAULT_PAGE_SIZE = 30;

    @Override
    public <T extends Serializable> T findOne(String namedQuery, List<Param> params, Class<T> type) {
        TypedQuery<T> q = getEntityManager().createNamedQuery(namedQuery, type);
        jpqlBuilder.setWhereParams(q, params);
        try {
            return q.getSingleResult();
        } catch (NoResultException e) {
            return null;
        }
    }

    public static final String EDITORROLE = "editor";

    /**
     * When overriding look into {@link Transactional} and roles to use, these
     * are not inherited. If you don't need security, override and use
     * {@link PermitAll}.
     *
     * @param <T>
     * @param t
     * @return
     */
    @Transactional
    @Override
    @RolesAllowed(value = {EDITORROLE})
    public <T extends Serializable> T create(T t) {
        getEntityManager().persist(t);
        return t;
    }

    /**
     *
     * When overriding look into {@link Transactional} and roles to use, these
     * are not inherited. If you don't need security, override and use
     * {@link PermitAll}.
     *
     * @param <T>
     * @param t
     * @return
     */
    @Transactional
    @Override
    @RolesAllowed(value = {EDITORROLE})
    public <T extends Serializable> T update(T t) {
        return getEntityManager().merge(t);
    }

    /**
     * When overriding look into {@link Transactional} and roles to use, these
     * are not inherited. If you don't need security, override and use
     * {@link PermitAll}.
     *
     * @param t
     */
    @Transactional
    @TransactionAttribute
    @Override
    @RolesAllowed(value = {EDITORROLE})
    public void delete(Serializable t) {
        getEntityManager().remove(getEntityManager().merge(t));
    }

    private List doFind(Query q, List<Param> params, Integer first, Integer max) {
        jpqlBuilder.setWhereParams(q, params);
        q.setFirstResult(first == null ? 0 : first);
        if (max != null) {
            if (max >= 0) {
                q.setMaxResults(max);
            }
        } else {
            q.setMaxResults(DEFAULT_PAGE_SIZE);
        }
        return q.getResultList();
    }

    private int doCount(Query q, List<Param> params) {
        jpqlBuilder.setWhereParams(q, params);
        return Integer.parseInt(String.valueOf(q.getSingleResult()));
    }

}
