package org.fryske_akademy.exist.jobs;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.exist.EXistException;
import org.exist.collections.Collection;
import org.exist.collections.IndexInfo;
import org.exist.collections.triggers.TriggerException;
import org.exist.dom.persistent.DocumentImpl;
import org.exist.security.PermissionDeniedException;
import org.exist.source.StringSource;
import org.exist.storage.DBBroker;
import org.exist.storage.SystemTask;
import org.exist.storage.txn.Txn;
import org.exist.util.Configuration;
import org.exist.util.FileInputSource;
import org.exist.util.LockException;
import org.exist.xmldb.XmldbURI;
import org.exist.xquery.CompiledXQuery;
import org.exist.xquery.XPathException;
import org.exist.xquery.XQuery;
import org.exist.xquery.XQueryContext;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Date;
import java.util.EnumSet;
import java.util.Properties;

/**
 * Sync files in a directory to a collection without recursing. Files that are new or newer than
 * the one in the target collection will be written to the collection. Files that are
 * not present in the source directory will be removed from the collection, this can be turned of
 * via {@link DataSyncTask#REMOVE_FROM_COLLECTION_PARAM}. By default xml files can be handled, for syncing
 * other types create a subclass and override {@link #storeInCollection(Path, XmldbURI, Collection, Txn, DBBroker)}.
 * After syncing cache is cleared to prevent problems, this can be turned off via {@link #CLEAR_CACHE_PARAM}.
 * Meant to be used as a start-up task, {@link DataSyncTaskCron} is meant to be scheduled as a cronjob.
 */
public class DataSyncTask implements SystemTask {


    private final static Logger LOG = LogManager.getLogger(DataSyncTask.class);

    /**
     * default data dir on filesystem (i.e. docker mount)
     */
    public static final String DATA_DIR = "/data";
    public static final String COLLECTION_PARAM = "collection";
    public static final String DATADIR_PARAM = "datadir";
    public static final String REMOVE_FROM_COLLECTION_PARAM = "removeNotInSource";
    /**
     * should be placed in the root of your jar
     */
    public static final String CLEAR_CACHE_XQ = "xquery version \"3.1\";\n" +
            "import module namespace cache = \"http://exist-db.org/xquery/cache\";\n" +
            "cache:clear()";
    public static final String CLEAR_CACHE_PARAM = "clearCache";

    private boolean removeMissingInSource;
    private String collection=null;
    private String dir=DATA_DIR;
    private boolean clearCache = true;
    private CompiledXQuery query;

    @Override
    public String getName() {
        return "Data Sync";
    }

    @Override
    public void configure(Configuration config, Properties properties) throws EXistException {
        collection=properties.getProperty(COLLECTION_PARAM);
        removeMissingInSource = Boolean.parseBoolean(properties.getProperty(REMOVE_FROM_COLLECTION_PARAM,"true"));
        dir=properties.getProperty(DATADIR_PARAM,DATA_DIR);
        clearCache=Boolean.parseBoolean(properties.getProperty(CLEAR_CACHE_PARAM,"true"));
    }

    @Override
    public void execute(DBBroker broker, Txn transaction) throws EXistException {
        try {
            if (collection==null) {
                throw new EXistException(String.format("You have to provide %s parameter in conf.xml",COLLECTION_PARAM));
            }
            LOG.info(String.format("start sync %s to %s",dir,collection));
            File dataDir = new File(dir);
            XmldbURI dataRoot = XmldbURI.xmldbUriFor(collection);
            Collection data = broker.getOrCreateCollection(transaction,dataRoot);
            if (removeMissingInSource) {
                // remove files not present in the source folder
                for (Collection.CollectionEntry ce : data.getEntries(broker)) {
                    final XmldbURI uri = ce.getUri();
                    if (dataDir.list(new FilenameFilter() {

                        @Override
                        public boolean accept(File file, String s) {
                            try {
                                return uri.endsWith("/" + s);
                            } catch (URISyntaxException e) {
                                return false;
                            }
                        }
                    }).length == 0) {
                        data.removeResource(transaction, broker, data.getDocument(broker, uri));
                    }
                }
            }
            Files.walkFileTree(dataDir.toPath(),
                    EnumSet.noneOf(FileVisitOption.class),
                    0,
                    new FileVisitor<Path>() {
                        @Override
                        public FileVisitResult preVisitDirectory(Path path, BasicFileAttributes basicFileAttributes) throws IOException {
                            return FileVisitResult.CONTINUE;
                        }

                        @Override
                        public FileVisitResult visitFile(Path path, BasicFileAttributes basicFileAttributes) throws IOException {
                            try {
                                XmldbURI docUri = XmldbURI.create(path.toFile().getName());
                                DocumentImpl document = data.getDocument(broker, dataRoot.append(path.toFile().getName()));
                                if (document == null ||
                                        document.getMetadata().getLastModified() < path.toFile().lastModified()) {
                                    storeInCollection(path, docUri, data, transaction, broker);
                                    if (LOG.isDebugEnabled()) LOG.debug((document==null?"created ":"updated ") + docUri);
                                } else {
                                    if (LOG.isDebugEnabled())
                                        LOG.debug(String.format("%s not updated, file older than doc: %s, %s",
                                                docUri,
                                                new Date(path.toFile().lastModified()),
                                                new Date(document.getMetadata().getLastModified())));
                                }
                            } catch (EXistException |PermissionDeniedException|SAXException|LockException e) {
                                throw new IOException(e);
                            }
                            return FileVisitResult.CONTINUE;
                        }

                        @Override
                        public FileVisitResult visitFileFailed(Path path, IOException e) throws IOException {
                            return FileVisitResult.CONTINUE;
                        }

                        @Override
                        public FileVisitResult postVisitDirectory(Path path, IOException e) throws IOException {
                            return FileVisitResult.CONTINUE;
                        }
                    });
            if (clearCache) {
            /*
            now clear cache, because of

WARN  (EmbeddedXMLStreamReader.java [verifyOriginNodeId]:239) - Expected node id 1.6.2.2.4.5.3, got 1.4.2.6.2.2.3.1; resyncing address
WARN  (TransactionManager.java [close]:409) - Transaction was not committed or aborted, auto aborting!
ERROR (EXistServlet.java [doPost]:488) - java.lang.NullPointerException

            and because cache may contain invalid data now

             */
                XQuery xQuery = new XQuery();
                if (query == null)
                    query = xQuery.compile(broker, new XQueryContext(broker.getDatabase()), new StringSource(CLEAR_CACHE_XQ));
                xQuery.execute(broker, query, null);
            }

            transaction.commit();

            LOG.info(String.format("end sync %s to %s",dir,collection));
        } catch (PermissionDeniedException | IOException | TriggerException | URISyntaxException | LockException | XPathException e) {
            throw new EXistException(e);
        }
    }

    /**
     * This method assumes xml is to be stored and uses {@link Collection#store(Txn, DBBroker, IndexInfo, InputSource)}
     * after validation to store the document.
     * @param fileToStore
     * @param documentInCollection
     * @param collection
     * @param transaction
     * @param broker
     * @throws EXistException
     * @throws PermissionDeniedException
     * @throws SAXException
     * @throws LockException
     * @throws IOException
     */
    protected void storeInCollection(Path fileToStore, XmldbURI documentInCollection, Collection collection, Txn transaction, DBBroker broker) throws EXistException, PermissionDeniedException, SAXException, LockException, IOException {
        IndexInfo indexInfo =
                collection.validateXMLResource(
                        transaction, broker, documentInCollection, new FileInputSource(fileToStore));
        collection.store(transaction, broker, indexInfo, new FileInputSource(fileToStore));
    }

    @Override
    public boolean afterCheckpoint() {
        return false;
    }
}
