package org.fryske_akademy.jsf;

/*-
 * #%L
 * guiCrudApi
 * %%
 * Copyright (C) 2018 Fryske Akademy
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

import java.io.IOException;
import java.io.Serializable;
import java.security.Principal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.PostConstruct;
import javax.faces.bean.SessionScoped;
import javax.faces.context.FacesContext;
import javax.faces.event.ValueChangeEvent;
import javax.servlet.ServletException;
import javax.servlet.http.Cookie;
import org.fryske_akademy.jsf.util.CookieHelper;
import org.fryske_akademy.jsf.util.JsfUtil;

/**
 * manages logout, internationalization and theming, configure this as a
 * managed-bean with name "sessionBean" in faces-config.xml, this way overriding
 * is more clear.
 *
 *
 *
 * @author eduard
 */
@SessionScoped
public class SessionBean implements Serializable {

    public static final String LANGUAGE_COOKIE = "language";
    public static final String THEME_COOKIE = "theme";

    private String language;
    private Locale locale;

    private static final Map<String, Locale> languageLocales = new HashMap<>(3);
    private static final List<String> languages = new ArrayList<>(3);
    private static final List<String> themes = new ArrayList<>(40);
    private String currentTheme = themes.get(2);

    static {
        themes.add("afterdark");
        themes.add("afternoon");
        themes.add("afterwork");
        themes.add("aristo");
        themes.add("black-tie");
        themes.add("blitzer");
        themes.add("bluesky");
        themes.add("bootstrap");
        themes.add("casablanca");
        themes.add("cupertino");
        themes.add("cruze");
        themes.add("dark-hive");
        themes.add("delta");
        themes.add("dot-luv");
        themes.add("eggplant");
        themes.add("excite-bike");
        themes.add("flick");
        themes.add("glass-x");
        themes.add("home");
        themes.add("hot-sneaks");
        themes.add("humanity");
        themes.add("le-frog");
        themes.add("midnight");
        themes.add("mint-choc");
        themes.add("overcast");
        themes.add("pepper-grinder");
        themes.add("redmond");
        themes.add("rocket");
        themes.add("sam");
        themes.add("smoothness");
        themes.add("south-street");
        themes.add("start");
        themes.add("sunny");
        themes.add("swanky-purse");
        themes.add("trontastic");
        themes.add("ui-darkness");
        themes.add("ui-lightness");
        themes.add("vader");
    }

    @PostConstruct
    private void init() {
        initLanguages();
        initFromCookies();
    }

    /**
     * list available themes
     * @return 
     */
    public List<String> getThemes() {
        return themes;
    }

    public String getCurrentTheme() {
        return currentTheme;
    }

    public void setCurrentTheme(String currentTheme) {
        this.currentTheme = currentTheme;
    }

    public void themeChanged(ValueChangeEvent event) {
        setCurrentTheme((String) event.getNewValue());
        CookieHelper.replaceCookie(THEME_COOKIE, currentTheme);
    }

    protected final void addLanguage(String name, String code, String region) {
        languageLocales.put(name, new Locale(code, region));
        if (!languages.contains(name)) {
            languages.add(name);
        }
    }

    /**
     * Called from @Postconstruct, sets current language and theme from a cookie or to a default
     * @see CookieHelper
     */
    protected void initFromCookies() {
        Cookie cookie = CookieHelper.getCookie(LANGUAGE_COOKIE);
        setLanguage(cookie != null ? cookie.getValue() : "Frisian");
        cookie = CookieHelper.getCookie(THEME_COOKIE);
        setCurrentTheme(cookie != null ? cookie.getValue() : "afterwork");
    }

    /**
     * Called from @Postconstruct,
     * initializes frisian, ducth and english.
     * Override this to support more / other languages
     */
    protected void initLanguages() {
        synchronized (languageLocales) {
            if (languageLocales.isEmpty()) {
                addLanguage("Frisian", "fy", "NL");
                addLanguage("Dutch", "nl", "NL");
                addLanguage("English", "en", "EN");
            }
        }
    }

    /**
     * use this for user choice (dropdown)
     *
     * @return
     */
    public List<String> getLanguages() {
        return languages;
    }

    /**
     * The current choice
     *
     * @return
     */
    public String getLanguage() {
        return language;
    }

    public void setLanguage(String language) {
        locale = languageLocales.get(language);
        FacesContext.getCurrentInstance()
                .getViewRoot().setLocale(locale);
        this.language = language;
    }

    /**
     * the current locale, use this in @locale in f:view
     *
     * @return
     */
    public Locale getLocale() {
        return locale;
    }

    public boolean mayEdit() {
        return JsfUtil.mayEdit();
    }

    /**
     * Call this when user chooses other language
     *
     * @param e
     */
    public void languageChanged(ValueChangeEvent e) {
        setLanguage((String) e.getNewValue());
        CookieHelper.replaceCookie(LANGUAGE_COOKIE, language);
    }

    public String getUser() {
        Principal p = JsfUtil.findPrinciple();
        return (p == null) ? "not logged in" : p.getName();
    }

    public void logout() {
        try {
            JsfUtil.logout();
            FacesContext.getCurrentInstance().getExternalContext().redirect(JsfUtil.contextPath() + "/login.xhtml");
        } catch (ServletException | IOException ex) {
            Logger.getLogger(SessionBean.class.getName()).log(Level.SEVERE, null, ex);
        }
    }

}
