/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.fryske_akademy.jsf.lazy;

/*-
 * #%L
 * guiCrudApi
 * %%
 * Copyright (C) 2018 Fryske Akademy
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.annotation.PostConstruct;
import javax.ejb.EJB;
import javax.faces.bean.SessionScoped;
import org.fryske_akademy.ejb.CrudReadService;
import org.fryske_akademy.ejb.JpqlBuilder.SORTORDER;
import org.fryske_akademy.ejb.ParamInfo;
import org.fryske_akademy.ejb.SortBuilder;
import org.fryske_akademy.jpa.AbstractEntity;
import org.fryske_akademy.jsf.AbstractCrudController;
import org.fryske_akademy.jsf.AbstractLazyController;
import org.fryske_akademy.jsf.Filtering;
import org.primefaces.model.LazyDataModel;
import org.primefaces.model.SortMeta;
import org.primefaces.model.SortOrder;

/**
 * Class to support generic lazy loading of data and generic filtering support,
 * declares @EJB CrudReadService.
 *
 * @author eduard
 */
@SessionScoped
public abstract class AbstractLazyModel<T extends AbstractEntity> extends LazyDataModel<T> implements Filtering<T> {

    @EJB
    private CrudReadService crudReadService;

    private final Class<T> clazz;

    /**
     * Used from {@link #load(int, int, java.util.List, java.util.Map) } to set {@link AbstractCrudController#setRememberTableState(boolean)
     * } to true and from {@link #select(java.lang.String, java.lang.String, int) }
     * to set {@link AbstractCrudController#setRememberTableState(boolean) } to
     * false.
     *
     * @return
     */
    protected abstract AbstractLazyController getLazyController();

    public AbstractLazyModel(Class<T> clazz) {
        this.clazz = clazz;
    }

    public CrudReadService getCrudReadService() {
        return crudReadService;
    }

    private final AtomicBoolean selecting = new AtomicBoolean();

    /**
     * map holding filter values, creates an entry with a null value if a key is
     * not yet in the map
     */
    private final Map<String, Object> filters = new HashMap<String, Object>(3) {
        @Override
        public Object get(Object key) {
            if (!containsKey(key)) {
                put(String.valueOf(key), null);
            }
            return super.get(key);
        }

    };

    /**
     * Calls {@link CrudReadService#find(java.io.Serializable, java.lang.Class)
     * } with Integer.valueOf(rowKey)
     *
     * @param rowKey
     * @return
     */
    @Override
    public T getRowData(String rowKey) {
        return crudReadService.find(Integer.valueOf(rowKey), clazz);
    }

    private List<T> filtered;

    /**
     * first time loading of data, calls {@link #setWrappedData(java.lang.Object)
     * } and {@link #setRowCount(int) }
     * with {@link CrudReadService#count(java.util.List, java.lang.Class) }.
     */
    @PostConstruct
    public void init() {
        List<T> load = load(0, getPageSize(), null, null, null);
        setWrappedData(load);
        setRowCount(crudReadService.count(null, clazz));
    }

    /**
     * removes filters on booelan fields with value false, so false show both
     * false and true and true only shows true
     *
     * @param filters
     * @return
     */
    protected List<ParamInfo> convertFilters(Map<String, Object> filters) {
        if (filters == null) {
            return null;
        }
        ParamInfo.ParamInfoBuilder builder = new ParamInfo.ParamInfoBuilder();
        for (Map.Entry<String, Object> p : filters.entrySet()) {
            addToParamBuilder(builder, p.getKey(), p.getValue());
        }
        return builder.build();
    }

    /**
     * adds {@link ParamInfo.ParamInfoBuilder#add(java.lang.String, java.lang.Object, boolean) } to
     * builder, uses {@link AbstractLazyController#isUseOr() }. Gives you control over parameter meta info for filter entries.
     * Doesn't ad boolean false values from filters, otherwise rows will always
     * be filtered on true or on false, true and false will never show both.
     *
     * @param key
     * @param value
     */
    protected void addToParamBuilder(ParamInfo.ParamInfoBuilder builder, String key, Object value) {
        if (!(value instanceof Boolean) || (Boolean) value) {
            builder.add(key, value, getLazyController().isUseOr());
        }
    }

    /**
     * Calls {@link #load(int, int, java.util.List, java.util.Map) }
     *
     * @param first
     * @param pageSize
     * @param sortField
     * @param sortOrder
     * @param filters
     * @return
     */
    @Override
    public final List<T> load(int first, int pageSize, String sortField, SortOrder sortOrder, Map<String, Object> filters) {
        List<SortMeta> s = new ArrayList<>(1);
        s.add(new SortMeta(null, sortField, sortOrder, null));
        return load(first, pageSize, s, filters);
    }

    /**
     * loads data via {@link CrudReadService#findPaged(int, int, java.util.Map, java.util.List, java.lang.Class)
     * }, {@link #convertFilters(java.util.Map) converts filters}. When
     * selecting, set {@link AbstractCrudController#setRememberTableState(boolean)
     * } to true to restore saving table state across views.
     *
     * @param first
     * @param pageSize
     * @param multiSortMeta
     * @param filters
     * @return
     */
    @Override
    public List<T> load(int first, int pageSize, List<SortMeta> multiSortMeta, Map<String, Object> filters) {
        if (selecting.getAndSet(false)) {
            getLazyController().setRememberTableState(true);
            return null;
        }
        SortBuilder sortBuilder = new SortBuilder();
        if (multiSortMeta != null) {
            multiSortMeta.forEach((h) -> {
                sortBuilder.add(h.getSortField(), convert(h.getSortOrder()));
            });
        }
        List<ParamInfo> convertFilters = convertFilters(filters);
        List<T> load = crudReadService.findPaged(first, pageSize, sortBuilder.build(), convertFilters, clazz);
        setWrappedData(load);
        setRowCount(crudReadService.count(convertFilters, clazz));
        if (filters != null && !filters.isEmpty()) {
            clear();
            this.filters.putAll(filters);
        }
        return load;
    }

    public static SORTORDER convert(SortOrder order) {
        if (order == null) {
            return null;
        }
        switch (order) {
            case ASCENDING:
                return SORTORDER.ASC;
            default:
                return SORTORDER.DESC;
        }
    }

    /**
     * Use this in primefaces EL expression for filteredValue
     *
     * @return
     */
    @Override
    public List<T> getFiltered() {
        return filtered;
    }

    @Override
    public void setFiltered(List<T> filtered) {
        this.filtered = filtered;
    }

    /**
     * used this in EL expression for filterValue:
     * #{controller.filters['filtername']}
     *
     * @return
     */
    @Override
    public Map<String, Object> getFilters() {
        return filters;
    }

    /**
     * filters data based on a key (column) and value by calling {@link #setFiltered(java.util.List)
     * }
     * with {@link #load(int, int, java.lang.String, org.primefaces.model.SortOrder, java.util.Map)
     * } as argument. POffset is set to 0. This method makes
     * sure load() isn't called the next time in the application lifecycle
     * because this would void the filtering. Set {@link AbstractCrudController#setRememberTableState(boolean)
     * } to false so saving table doesn't interfere with the selecting
     * mechanism.
     *
     * @param key
     * @param value
     */
    @Override
    public final void select(String key, String value, int maxResults) {
        selecting.set(false);
        Map<String, Object> f = new HashMap<String, Object>(1);
        f.put(key, value);
        setFiltered(load(0, maxResults, null, null, f));
        getLazyController().setRememberTableState(false);
        selecting.set(true);
    }

    @Override
    public void clear() {
        filters.clear();
    }

}
