/*
 * Copyright 2018 Fryske Akademy.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.fryske_akademy.jsf;


import javax.faces.context.FacesContext;
import javax.faces.event.PhaseEvent;
import javax.faces.event.PhaseId;
import javax.faces.event.PhaseListener;
import javax.servlet.DispatcherType;
import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * redirect when ajax request encounters session timeout. Typically this is how it works:
 * <ol>
 * <li>ajax request encounters invalid session</li>
 * <li>this listener detects {@link DispatcherType#FORWARD} to {@link #getLoginPath() }</li>
 * <li>{@link #handleAjaxAfterTimeout(javax.faces.context.FacesContext, javax.servlet.http.HttpServletRequest) } is called</li>
 * <li>In this method a redirect is done to one of your protected pages</li>
 * <li>your security setup redirects to the login page</li>
 * <li>after login you land on the page from step 4</li>
 * </ol>
 * declare this listener in your faces-config
 * @author eduard
 */
public class AjaxSessionTimeoutListener implements PhaseListener {
    
    private static final Logger LOGGER = Logger.getLogger(AjaxSessionTimeoutListener.class.getName());
    
    public static final String LOGINPATH = "/login.xhtml";

    /**
     * @return {@link #LOGINPATH}
     */
    protected String getLoginPath() {
        return LOGINPATH;
    }

    @Override
    public void beforePhase(final PhaseEvent event) {
        final FacesContext facesContext = event.getFacesContext();
        if (!facesContext.getPartialViewContext().isAjaxRequest() || facesContext.getRenderResponse()) {
            if (facesContext.getRenderResponse() && LOGGER.isLoggable(Level.FINE)) {
                final HttpServletRequest request = HttpServletRequest.class.cast(facesContext.getExternalContext().getRequest());
                LOGGER.fine(String.format("response rendered, not handling %s", request.getServletPath()));
            }
            return;
        }

        final HttpServletRequest request = HttpServletRequest.class.cast(facesContext.getExternalContext().getRequest());
        if (request.getDispatcherType() == DispatcherType.FORWARD && getLoginPath().equals(request.getServletPath())) { // isLoginRedirection()
            try {
                handleAjaxAfterTimeout(facesContext, request);
            } catch (final IOException e) {
                LOGGER.log(Level.SEVERE, "unable to redirect", e);
            }
        }
    }
    
    /**
     * This implementation redirects to the context path root
     * @param facesContext
     * @param request the original ajax request
     * @throws java.io.IOException
     */
    protected void handleAjaxAfterTimeout(FacesContext facesContext, HttpServletRequest request) throws IOException {
        facesContext.getExternalContext().redirect(request.getContextPath());
    }

    @Override
    public void afterPhase(final PhaseEvent event) {
    }

    @Override
    public PhaseId getPhaseId() {
        return PhaseId.RESTORE_VIEW;
    }
}
