/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.fryske_akademy.jsf.lazy;

/*-
 * #%L
 * guiCrudApi
 * %%
 * Copyright (C) 2018 Fryske Akademy
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.faces.bean.SessionScoped;
import org.fryske_akademy.ejb.CrudReadService;
import org.fryske_akademy.ejb.CrudReadService.SORTORDER;
import org.fryske_akademy.ejb.Param;
import org.fryske_akademy.ejb.Param.Builder;
import org.fryske_akademy.jpa.EntityInterface;
import org.fryske_akademy.jsf.AbstractEntityController;
import org.fryske_akademy.jsf.AbstractLazyController;
import org.fryske_akademy.jsf.Filtering;
import org.fryske_akademy.jsf.util.JsfUtil;
import org.primefaces.model.LazyDataModel;
import org.primefaces.model.SortMeta;
import org.primefaces.model.SortOrder;

/**
 * Class to support generic lazy loading of data and generic filtering support.
 * The prevered usage is to create a managed bean of a subclass and provide that
 * as an argument to
 * {@link AbstractLazyController#AbstractLazyController(java.lang.Class, org.fryske_akademy.jsf.lazy.AbstractLazyModel) }
 * using {@link JsfUtil#findInContext(java.lang.String, java.lang.Class) }.
 *
 * @author eduard
 */
@SessionScoped
public abstract class AbstractLazyModel<T extends EntityInterface> extends LazyDataModel<T> implements Filtering<T> {

    private final Class<T> clazz;
    private CrudReadService crudReadService;
    private boolean useOr;
    private boolean syntaxInvalue = true;

    private AbstractLazyController abstractLazyController;

    public AbstractLazyController getLazyController() {
        return abstractLazyController;
    }

    /**
     * Called in @Postconstruct of {@link AbstractLazyController}
     *
     * @param lazyController
     */
    public final void setLazyController(AbstractLazyController lazyController) {
        if (abstractLazyController != null) {
            abstractLazyController = lazyController;
            crudReadService = lazyController.getCrudWriteService();
        } else {
            throw new IllegalStateException("lazy controller already wired");
        }
    }

    public AbstractLazyModel(Class<T> clazz) {
        this.clazz = clazz;
    }

    public final CrudReadService getCrudReadService() {
        if (crudReadService == null) {
            throw new IllegalStateException("Please inject crudReadService via setLazyController()");
        }
        return crudReadService;
    }

    private final AtomicBoolean selecting = new AtomicBoolean();
    private final AtomicBoolean tableState = new AtomicBoolean();

    /**
     * map holding filter values, creates an entry with a null value if a key is
     * not yet in the map
     */
    private final Map<String, Object> filters = new HashMap<String, Object>(3) {
        @Override
        public Object get(Object key) {
            if (!containsKey(key)) {
                put(String.valueOf(key), null);
            }
            return super.get(key);
        }

    };

    /**
     * Calls {@link CrudReadService#find(java.io.Serializable, java.lang.Class)
     * } with Integer.valueOf(rowKey)
     *
     * @param rowKey
     * @return
     */
    @Override
    public T getRowData(String rowKey) {
        return crudReadService.find(Integer.valueOf(rowKey), clazz);
    }

    private List<T> filtered;

    /**
     * Called from PostConstruct. First time loading of data, calls {@link #setWrappedData(java.lang.Object)
     * } and {@link #setRowCount(int) }
     * with {@link CrudReadService#countDynamic(java.util.List, java.lang.Class)
     * }.
     */
    protected void init() {
        List<T> load = load(0, getPageSize(), null, null, null);
        setWrappedData(load);
        setRowCount(crudReadService.countDynamic(null, clazz));
    }

    /**
     * creates a {@link Builder }
     * using {@link AbstractLazyController#isSyntaxInvalue() } and
     * {@link Builder#DEFAULT_MAPPING}.
     *
     * @return
     */
    protected Param.Builder initParamBuilder() {
        return new Param.Builder(isSyntaxInvalue(), Param.Builder.DEFAULT_MAPPING);
    }

    /**
     * Call {@link #initParamBuilder() } and call {@link #addToParamBuilder(org.fryske_akademy.ejb.Param.Builder, java.lang.String, java.lang.Object)
     * } for each entry.
     *
     * @param filters
     * @return
     */
    protected final List<Param> convertFilters(Map<String, Object> filters) {
        if (filters == null) {
            return null;
        }
        Param.Builder builder = initParamBuilder();
        for (Map.Entry<String, Object> p : filters.entrySet()) {
            addToParamBuilder(builder, p.getKey(), p.getValue());
        }
        return builder.build();
    }

    /**
     * adds {@link Builder#add(java.lang.String, java.lang.Object, boolean) } to
     * builder, uses {@link AbstractLazyController#isUseOr() }. Gives you
     * control over parameter meta info for filter entries. Doesn't add boolean
     * false values from filters, otherwise rows will always be filtered on true
     * or on false, true and false will never show both.
     *
     * @param builder
     * @param key
     * @param value
     */
    protected void addToParamBuilder(Param.Builder builder, String key, Object value) {
        if (!(value instanceof Boolean) || (Boolean) value) {
            builder.add(key, value, isUseOr());
        }
    }

    /**
     * Calls {@link #load(int, int, java.util.List, java.util.Map) }
     *
     * @param first
     * @param pageSize
     * @param sortField
     * @param sortOrder
     * @param filters
     * @return
     */
    @Override
    public final List<T> load(int first, int pageSize, String sortField, SortOrder sortOrder, Map<String, Object> filters) {
        List<SortMeta> s = new ArrayList<>(1);
        s.add(new SortMeta(null, sortField, sortOrder, null));
        return load(first, pageSize, s, filters);
    }

    /**
     * loads data via {@link CrudReadService#findDynamic(java.lang.Integer, java.lang.Integer, java.util.Map, java.util.List, java.lang.Class)
     * }, {@link #convertFilters(java.util.Map) converts filters}. When
     * {@link #select(java.lang.String, java.lang.String)  selecting}, set {@link AbstractEntityController#setRememberTableState(boolean)
     * } to its original value.
     *
     * @param first
     * @param pageSize
     * @param multiSortMeta
     * @param filters
     * @return
     */
    @Override
    public final List<T> load(int first, int pageSize, List<SortMeta> multiSortMeta, Map<String, Object> filters) {
        SORTORDER.Builder sortBuilder = new SORTORDER.Builder();
        if (multiSortMeta != null) {
            multiSortMeta.forEach((h) -> {
                sortBuilder.add(h.getSortField(), convert(h.getSortOrder()));
            });
        }
        List<Param> convertFilters = convertFilters(filters);
        List<T> load = crudReadService.findDynamic(first, pageSize, sortBuilder.build(), convertFilters, clazz);
        setWrappedData(load);
        setRowCount(crudReadService.countDynamic(convertFilters, clazz));
        if (selecting.getAndSet(false)) {
            abstractLazyController.setRememberTableState(tableState.get());
        }
        return load;
    }

    public static SORTORDER convert(SortOrder order) {
        if (order == null) {
            return null;
        }
        switch (order) {
            case ASCENDING:
                return SORTORDER.ASC;
            default:
                return SORTORDER.DESC;
        }
    }

    /**
     * Use this in primefaces EL expression for filteredValue
     *
     * @return
     */
    @Override
    public List<T> getFiltered() {
        return filtered;
    }

    @Override
    public void setFiltered(List<T> filtered) {
        this.filtered = filtered;
    }

    /**
     * used this in EL expression for filterValue:
     * #{controller.filters['filtername']}
     *
     * @return
     */
    @Override
    public Map<String, Object> getFilters() {
        return filters;
    }

    /**
     * Selects specific data based on a key (filter) and value by putting only
     * this key/value pair in filters. The next time 
     * {@link #load(int, int, java.lang.String, org.primefaces.model.SortOrder, java.util.Map) }
     * is called the filter is used. Set {@link AbstractEntityController#setRememberTableState(boolean)
     * } to false, because it would void the filtering, the value is restored in
     * the next load.
     *
     * @param key
     * @param value
     */
    public final void select(String key, String value) {
        clear().add(key, value);
        if (!selecting.get()) {
            tableState.set(abstractLazyController.isRememberTableState());
        }
        abstractLazyController.setRememberTableState(false);
        selecting.set(true);
    }

    @Override
    public Filtering<T> add(String key, Object value) {
        filters.put(key, value);
        return this;
    }

    @Override
    public Filtering<T> clear() {
        filters.clear();
        return this;
    }

    public void setSyntaxInvalue(boolean syntaxInvalue) {
        this.syntaxInvalue = syntaxInvalue;
    }

    /**
     * when true use or when several parameters are given
     *
     * @see Param#getAndOr()
     * @return
     */
    public boolean isUseOr() {
        return useOr;
    }

    public void setUseOr(boolean useOr) {
        this.useOr = useOr;
    }

    /**
     * when true support syntax in parameter value
     *
     * @see Builder
     * @return
     */
    public boolean isSyntaxInvalue() {
        return syntaxInvalue;
    }

}
