/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.fryske_akademy.jsf;

/*-
 * #%L
 * guiCrudApi
 * %%
 * Copyright (C) 2018 Fryske Akademy
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR STATES OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import javax.faces.model.DataModel;
import org.fryske_akademy.ejb.Auditing;
import org.fryske_akademy.jpa.EntityInterface;
import org.fryske_akademy.jpa.RevInfo;
import org.fryske_akademy.jsf.util.JsfUtil;
import org.hibernate.envers.RevisionType;
import org.primefaces.component.datatable.DataTable;
import org.primefaces.event.RowEditEvent;
import org.primefaces.event.SelectEvent;

/**
 *
 * @author eduard
 */
public abstract class AbstractEntityController<T extends EntityInterface, S extends Auditing> extends AbstractController<S> {

    private static final Logger LOGGER = Logger.getLogger(AbstractEntityController.class.getName());

    private T selected;
    private T newEntity;
    protected final Class<T> clazz;

    private boolean rememberTableState = true;

    /**
     * remember sorting, filtering, column order, selection, column widths, page
     * number (@multiViewState). NOTE this may overwrite filter value set from {@link #filterAndRedirect(org.fryske_akademy.jsf.AbstractEntityController, java.lang.String, java.lang.String, java.lang.String)
     * }
     * subclasses may therefore choose to check if a request parameter "state" equals "filtering".
     *
     * @return
     */
    public boolean isRememberTableState() {
        return rememberTableState;
    }

    public void setRememberTableState(boolean rememberTableState) {
        this.rememberTableState = rememberTableState;
    }

    public AbstractEntityController(Class<T> clazz) {
        this.clazz = clazz;
    }

    public T getSelected() {
        return selected;
    }

    public void setSelected(T selected) {
        this.selected = selected;
    }

    public T getNewEntity() {
        return newEntity;
    }

    public void setNewEntity(T newEntity) {
        this.newEntity = newEntity;
    }

    public void onRowSelect(SelectEvent event) {
        if (event != null) {
            setSelected((T) event.getObject());
        }
    }

    /**
     * creates a new entity and calls {@link #fillNew(org.fryske_akademy.jpa.EntityInterface)
     * }
     *
     * @throws InstantiationException
     * @throws IllegalAccessException
     */
    public final void prepareCreate() throws InstantiationException, IllegalAccessException {
        newEntity = clazz.newInstance();
        fillNew(newEntity);
    }

    /**
     * calls {@link #persist(org.fryske_akademy.jpa.EntityInterface, org.fryske_akademy.jsf.util.JsfUtil.PersistAction, java.lang.String)
     * }, the returned entity will be the same as {@link #getNewEntity() }.
     */
    public T create() {
        return persist(newEntity, JsfUtil.PersistAction.CREATE, JsfUtil.getLocaleBundle(getBundleName()).getString("Created"));
    }

    /**
     * override if you want to use different bundle name then
     * {@link #DEFAULT_BUNDLE_NAME}
     *
     * @return
     */
    protected String getBundleName() {
        return DEFAULT_BUNDLE_NAME;
    }
    public static final String DEFAULT_BUNDLE_NAME = "/Bundle";

    public abstract Filtering<T> getFiltering();

    /**
     * return the value os a filter, often this will be a String.
     * @param key
     * @return 
     */
    public Object getFilterValue(String key) {
        return getFiltering().getFilters().get(key);
    }
    
    public String getFilterString(String key) {
        Object value = getFiltering().getFilters().get(key);
        return value==null?null:String.valueOf(value);
    }

    /**
     * active filters on a page, use this as primefaces @filterValue: filterValue="#{ctlr.filters['id']}"
     * @param key
     * @return 
     */
    public Map<String,Object> getFilters(String key) {
        return getFiltering().getFilters();
    }

    /**
     * Navigate (i.e. in a datatable) to the page containing the argument
     * entity. Implementors can call Filtering.clear().add(java.lang.String,
     * java.lang.Object) and then for example redirect and use viewParam
     *
     * @param entity
     * @return the action to perform
     */
    public abstract String gotoPageContaining(T entity);

    /**
     * implementors can fill the new object with values from selected
     *
     * @param newEntity
     * @param selected
     */
    protected abstract void fillCopy(T newEntity, T selected);

    /**
     * empty method, called from {@link #prepareCreate() }
     *
     * @param entity
     */
    protected void fillNew(T entity) {
    }

    /**
     * copy an entity, call {@link #fillCopy(org.fryske_akademy.jpa.EntityInterface, org.fryske_akademy.jpa.EntityInterface)
     * }
     *
     * @throws InstantiationException
     * @throws IllegalAccessException
     */
    public final void copy() throws InstantiationException, IllegalAccessException {
        if (selected != null) {
            T l = clazz.newInstance();
            fillCopy(l, getSelected());
            newEntity = l;
        }
    }

    /**
     * Calls {@link #persist(org.fryske_akademy.jpa.EntityInterface, org.fryske_akademy.jsf.util.JsfUtil.PersistAction, java.lang.String)
     * } and {@link #refreshRow(org.primefaces.event.RowEditEvent, org.fryske_akademy.jpa.EntityInterface, boolean)
     * } with the up to date entity and true.
     *
     * @param editEvent
     * @return the updated entity
     */
    public T update(RowEditEvent editEvent) {
        T t = persist((T) editEvent.getObject(), JsfUtil.PersistAction.UPDATE, JsfUtil.getLocaleBundle(getBundleName()).getString("Updated"));
        refreshRow(editEvent, t, true);
        return t;
    }

    /**
     * Attempts to show data in sync with the database after an update or cancel
     * edit. This implementation works for DataTables with a List or a DataModel
     * wrapping a List as value.
     *
     * @param editEvent
     * @param t
     * @param update when
     */
    protected void refreshRow(RowEditEvent editEvent, T t, boolean update) {
        if (editEvent.getComponent() instanceof DataTable) {
            DataTable dt = (DataTable) editEvent.getComponent();
            Object value = dt.getValue();
            List rows = value instanceof List ? (List) value : null;
            if (value instanceof DataModel && ((DataModel) value).getWrappedData() instanceof List) {
                rows = (List) ((DataModel) value).getWrappedData();
            }
            if (rows != null && dt.getRowIndex() > -1 && t.getId().equals(((T) dt.getRowData()).getId())) {
                int rowIndex = value instanceof DataModel ? ((DataModel)value).getRowIndex() : dt.getRowIndex();
                if (update) {
                    rows.set(rowIndex, t);
                } else {
                    rows.set(rowIndex, getCrudWriteService().find(((T) dt.getRowData()).getId(), clazz));
                }
            } else {
                LOGGER.warning(String.format("Cannot refresh datatable row: %s", dt.getRowIndex()));
            }
        }
    }

    /**
     * Calls {@link #refreshRow(org.primefaces.event.RowEditEvent, org.fryske_akademy.jpa.EntityInterface, boolean)
     * } with the canceled entity from the event and false.
     *
     * @param editEvent
     */
    public void cancelRow(RowEditEvent editEvent) {
        refreshRow(editEvent, (T) editEvent.getObject(), false);
    }

    public void destroy() {
        persist(selected, JsfUtil.PersistAction.DELETE, JsfUtil.getLocaleBundle(getBundleName()).getString("Deleted"));
        if (!JsfUtil.isValidationFailed()) {
            selected = null; // Remove selection
        }
    }

    /**
     * returns username, revision date and {@link RevisionType} for the last
     * revision found
     *
     * @see Auditing#getRevisionInfo(java.io.Serializable, java.lang.Integer,
     * java.lang.Class)
     * @param entity
     * @return
     */
    public String getLastChangedInfo(T entity) {
        List<RevInfo<T>> revisions = getCrudWriteService().getRevisionInfo(entity.getId(), 1, clazz);
        if (revisions.isEmpty()) {
            return "no changes";
        }
        RevInfo<T> rev = revisions.get(0);
        return rev.getRevisionInfo().getUsername() + " at " + rev.getRevisionInfo().getRevisionDate() + " (" + rev.getType() + ")";
    }

    public List<RevInfo<T>> getLastChanged(T entity, int max) {
        return getCrudWriteService().getRevisionInfo(entity, max, clazz);
    }

    /**
     * clears all filters, adds the key/value to it and returns your action with
     * {@link #FACESREDIRECTTRUE} and "state=filtering" appended. NOTE it may be necessary
     * to override {@link #isRememberTableState() } when the value set here is overwritten.
     *
     * @param controller
     * @param key
     * @param value
     * @param action
     * @return
     */
    protected String filterAndRedirect(AbstractEntityController controller, String key, String value, String action) {
        controller.getFiltering().clear().add(key, value);
        // redirect
        return action + FACESREDIRECTTRUE + "&" + STATE + "=" + FILTERING;
    }
    /**
     * will be appended as param key in {@link #filterAndRedirect(org.fryske_akademy.jsf.AbstractEntityController, java.lang.String, java.lang.String, java.lang.String)
     * }
     */
    protected static final String STATE = "state";

    /**
     * will be appended as param value in {@link #filterAndRedirect(org.fryske_akademy.jsf.AbstractEntityController, java.lang.String, java.lang.String, java.lang.String)
     * }
     */
    protected static final String FILTERING = "filtering";
}
