package org.fryske_akademy.jsf.util;

/*-
 * #%L
 * guiCrudApi
 * %%
 * Copyright (C) 2018 Fryske Akademy
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */
import java.security.Principal;
import java.util.List;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.security.RolesAllowed;
import javax.faces.application.FacesMessage;
import javax.faces.component.UIComponent;
import javax.faces.context.ExternalContext;
import javax.faces.context.FacesContext;
import javax.faces.convert.Converter;
import javax.faces.model.SelectItem;
import javax.security.enterprise.SecurityContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import org.fryske_akademy.Util;
import org.fryske_akademy.ejb.AbstractCrudService;

public class JsfUtil {

    private static final Logger LOGGER = Logger.getLogger(JsfUtil.class.getName());

    /**
     * Calls {@link FacesContext#validationFailed() }, {@link #deepestCause(java.lang.Throwable)
     * } and {@link #addErrorMessage(java.lang.String) } with the prefix and the
     * localized message from the exception. Logs the exception at level
     * WARNING.
     *
     * @param ex
     * @param prefix
     */
    public static void handleException(Exception ex, String prefix) {
        if (LOGGER.isLoggable(Level.WARNING)) {
            LOGGER.log(Level.WARNING, prefix, ex);
        }
        FacesContext.getCurrentInstance().validationFailed();
        JsfUtil.addErrorMessage(prefix + Util.deepestCause(ex).getLocalizedMessage());
    }

    public static SelectItem[] getSelectItems(List entities, boolean selectOne) {
        int size = selectOne ? entities.size() + 1 : entities.size();
        SelectItem[] items = new SelectItem[size];
        int i = 0;
        if (selectOne) {
            items[i++] = new SelectItem("", "---");
        }
        for (Object x : entities) {
            items[i++] = new SelectItem(x, x.toString());
        }
        return items;
    }

    public static boolean isValidationFailed() {
        return FacesContext.getCurrentInstance().isValidationFailed();
    }

    public static void addErrorMessage(Exception ex, String defaultMsg) {
        String msg = ex.getLocalizedMessage();
        if (msg != null && msg.length() > 0) {
            addErrorMessage(msg);
        } else {
            addErrorMessage(defaultMsg);
        }
    }

    public static void addErrorMessages(List<String> messages) {
        for (String message : messages) {
            addErrorMessage(message);
        }
    }

    /**
     * uses null as clientId
     *
     * @param msg
     */
    public static void addErrorMessage(String msg) {
        String cid = null;
        addErrorMessage(cid, msg);
    }

    public static void addErrorMessage(String clientId, String msg) {
        FacesMessage facesMsg = new FacesMessage(FacesMessage.SEVERITY_ERROR, msg, msg);
        FacesContext.getCurrentInstance().addMessage(clientId, facesMsg);
    }

    /**
     * Uses successInfo as client id
     *
     * @param msg
     */
    public static void addSuccessMessage(String msg) {
        addSuccessMessage("successInfo", msg);
    }

    public static void addSuccessMessage(String clientId, String msg) {
        FacesMessage facesMsg = new FacesMessage(FacesMessage.SEVERITY_INFO, msg, msg);
        FacesContext.getCurrentInstance().addMessage(clientId, facesMsg);
    }

    public static String getRequestParameter(String key) {
        return FacesContext.getCurrentInstance().getExternalContext().getRequestParameterMap().get(key);
    }

    public static Object getObjectFromRequestParameter(String requestParameterName, Converter converter, UIComponent component) {
        String theId = JsfUtil.getRequestParameter(requestParameterName);
        return converter.getAsObject(FacesContext.getCurrentInstance(), component, theId);
    }

    /**
     * Calls {@link #findInContext(java.lang.String, java.lang.Class) } with clazz.simpleName, first letter lowerCase
     * @param <T>
     * @param clazz
     * @return 
     */
    public static <T> T findInContext(Class<T> clazz) {
        return findInContext(Character.toLowerCase(clazz.getSimpleName().charAt(0)) + clazz.getSimpleName().substring(1), clazz);
    }
    
    public static <T> T findInContext(String name, Class<T> clazz) {
        FacesContext facesContext = FacesContext.getCurrentInstance();
        return (T) facesContext.getApplication().getELResolver().
                getValue(facesContext.getELContext(), null, name);
    }

    public static <T> T findOrCreateInContext(String name, Class<T> clazz) {
        FacesContext facesContext = FacesContext.getCurrentInstance();
        return (T) facesContext.getApplication().evaluateExpressionGet(facesContext, name, clazz);
    }

    public static void registerConverter(String id, Class clazz) {
        FacesContext facesContext = FacesContext.getCurrentInstance();
        facesContext.getApplication().addConverter(id, clazz.getName());
    }

    public static Converter getConverter(String id, Class clazz) {
        FacesContext facesContext = FacesContext.getCurrentInstance();
        return facesContext.getApplication().createConverter(id);
    }
    
    public static Converter getConverter(Class clazz) {
        FacesContext facesContext = FacesContext.getCurrentInstance();
        return facesContext.getApplication().createConverter(clazz);
    }

    /**
     * context path plus servlet path
     *
     * @return
     */
    public static String fullServletPath() {
        ExternalContext externalContext = FacesContext.getCurrentInstance().getExternalContext();
        return externalContext.getRequestContextPath() + externalContext.getRequestServletPath();
    }

    public static String contextPath() {
        ExternalContext externalContext = FacesContext.getCurrentInstance().getExternalContext();
        return externalContext.getRequestContextPath();
    }

    public static enum PersistAction {
        CREATE,
        DELETE,
        UPDATE
    }

    /**
     * Calls {@link #split(java.lang.String, int, boolean) } with false.
     *
     * @param toSplit
     * @param index
     * @return
     */
    @Deprecated
    public static String split(String toSplit, int index) {
        return Util.split(toSplit, index);
    }

    /**
     * Call {@link #split(java.lang.String, java.lang.String, int, boolean) }
     * with ": ?" as regex.
     *
     * @param toSplit
     * @param index
     * @param ignorePatternAfterIndex
     * @return
     */
    @Deprecated
    public static String split(String toSplit, int index, boolean ignorePatternAfterIndex) {
        return Util.split(toSplit, index, ignorePatternAfterIndex);
    }

    /**
     * split a string on regex and return the requested index, or null. Can be
     * used to find an entity using a readable user representation of an entity
     * such as "netherlands: franeker", or "netherlands: franeker: center".
     *
     * @param toSplit
     * @param index
     * @param ignorePatternAfterIndex when true, occurring ": ?" patterns after
     * index will be ignored, so the return value will be the rest of the input
     * string.
     * @return
     */
    @Deprecated
    public static String split(String toSplit, String regex, int index, boolean ignorePatternAfterIndex) {
        return Util.split(toSplit, regex, index, ignorePatternAfterIndex);
    }

    @Deprecated
    public static Throwable deepestCause(Throwable t) {
        return Util.deepestCause(t);
    }

    /**
     * @deprecated inject {@link SecurityContext}
     * @return 
     */
    public static Principal findPrinciple() {
        return FacesContext.getCurrentInstance().getExternalContext().getUserPrincipal();
    }

    public static void logout() throws ServletException {
        FacesContext.getCurrentInstance().getExternalContext().invalidateSession();
        ((HttpServletRequest) FacesContext.getCurrentInstance().getExternalContext().getRequest()).logout();
    }

    private static final ResourceBundle rootBundle = ResourceBundle.getBundle("EMPTY", Locale.ROOT);

    /**
     * return a resource for the current locale (from faces context or default locale), or an empty bundle
     *
     * @param bundle
     * @return
     */
    public static ResourceBundle getLocaleBundle(String bundle) {
        Locale loc = FacesContext.getCurrentInstance()!=null?FacesContext.getCurrentInstance().getViewRoot().getLocale():Locale.getDefault();
        try {
            return ResourceBundle.getBundle(bundle, loc);
        } catch (MissingResourceException e) {
            LOGGER.warning("resource not found: " + bundle + ", returning EMPTY");
            return rootBundle;
        }

    }

    /**
     * Calls {@link #mayEdit(java.lang.String) } with
     * {@link AbstractCrudService#EDITORROLE}.
     *
     * @return
     */
    public static boolean mayEdit() {
        return mayEdit(AbstractCrudService.EDITORROLE);
    }

    /**
     * checks if user has the role, which allows editing. Depends on your {@link RolesAllowed} setup.
     * 
     * @deprecated inject {@link SecurityContext} and use that
     *
     * @return
     */
    public static boolean mayEdit(String role) {
        return FacesContext.getCurrentInstance().getExternalContext().isUserInRole(role);
    }
}
