/*
 * Copyright 2018 Fryske Akademy.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.fryske_akademy;

import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Scanner;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.ejb.EJBException;
import javax.enterprise.context.spi.CreationalContext;
import javax.enterprise.inject.spi.Bean;
import javax.enterprise.inject.spi.BeanManager;
import javax.enterprise.inject.spi.CDI;
import javax.enterprise.util.AnnotationLiteral;

/**
 *
 * @author eduard
 */
public class Util {

    /**
     * parse a string containing enums the form "enum simple class"."enum name"
     * separated by space, look for the right setter in the Object class and
     * call the setter. Case is ignored.
     *
     * @param inputString space separated inputStrng, containing "enum simple
     * class"."enum name"
     * @param objectWithEnums
     * @throws IllegalArgumentException when no setter has been called for an
     * enum string
     */
    public static <T> T setEnumsFromString(String inputString, T objectWithEnums) {
        new Scanner(inputString).forEachRemaining((t) -> {
            boolean set = false;
            if (inputString.indexOf(".") > 0) {
                String enumName = t.substring(0, t.indexOf('.'));
                String enumValue = t.substring(t.indexOf('.') + 1);
                for (Method m : objectWithEnums.getClass().getMethods()) {
                    if (m.getName().equalsIgnoreCase("set" + enumName)) {
                        Class clazz = m.getParameterTypes()[0];
                        for (Enum e : (Enum[]) clazz.getEnumConstants()) {
                            if (e.name().equalsIgnoreCase(enumValue)) {
                                try {
                                    m.invoke(objectWithEnums, e);
                                    set = true;
                                } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException ex) {
                                    throw new EJBException(ex);
                                }
                            }
                        }
                    }
                }
            }
            if (!set) {
                throw new IllegalArgumentException(t + " not found in " + objectWithEnums);
            }
        });
        return objectWithEnums;
    }

    /**
     * look for a setter ignoring case for a string in the form "enum simple
     * class"."enum name", return true if it is found.
     *
     * @param t
     * @param objectWithEnums
     * @return
     */
    public static boolean hasSetterForEnumString(String t, Object objectWithEnums) {
        if (t.indexOf(".") > 0) {
            String enumName = t.substring(0, t.indexOf('.'));
            String enumValue = t.substring(t.indexOf('.') + 1);
            for (Method m : objectWithEnums.getClass().getMethods()) {
                if (m.getName().equalsIgnoreCase("set" + enumName)) {
                    Class clazz = m.getParameterTypes()[0];
                    for (Enum e : (Enum[]) clazz.getEnumConstants()) {
                        if (e.name().equalsIgnoreCase(enumValue)) {
                            return true;
                        }
                    }
                }
            }
        }
        return false;
    }

    /**
     * Look for a setter for the enum in the object, call it and return the
     * object.
     *
     * @param <T>
     * @param e
     * @param objectWithEnums
     * @return
     * @throws IllegalArgumentException when no setter has been called for the
     * enum
     */
    public static <T> T setEnum(Enum e, T objectWithEnums) {
        for (Method m : objectWithEnums.getClass().getMethods()) {
            if (m.getName().equals("set" + e.getClass().getSimpleName())) {
                try {
                    m.invoke(objectWithEnums, e);
                    return objectWithEnums;
                } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException ex) {
                    throw new EJBException(ex);
                }
            }
        }
        throw new IllegalArgumentException(e + " not found in " + objectWithEnums);
    }

    /**
     * look for a method returning an enum, call it, if the value isn't null
     * return it.
     *
     * @param objectWithEnums
     * @return
     */
    public static Enum getEnumWithValue(Object objectWithEnums) {
        for (Method m : objectWithEnums.getClass().getMethods()) {
            if (m.getReturnType().isEnum() && m.getParameterCount() == 0) {
                try {
                    Object o = m.invoke(objectWithEnums);
                    if (o != null) {
                        return (Enum) o;
                    }
                } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException ex) {
                    throw new EJBException(ex);
                }
            }
        }
        return null;
    }

    /**
     * look for all methods whose return type is an enum and add all constants
     * to the result.
     *
     * @param clazz
     * @return
     */
    public static List<Enum> listEnums(Class clazz) {
        List<Enum> rv = new ArrayList<>(100);
        for (Method m : clazz.getMethods()) {
            Class<?> c = m.getReturnType();
            if (c.isEnum()) {
                Enum[] e = (Enum[]) c.getEnumConstants();
                rv.addAll(Arrays.asList(e));
            }
        }
        return rv;
    }

    /**
     * Look for methods with return type enum, whose simple class is the first
     * part of the argument string, look for the enum constant whose name is the
     * last part of the argument string. Case is ignored.
     *
     * @param clazz
     * @param e string in the form "enum simple class"."enum name"
     * @return
     */
    public static Enum findInClass(Class clazz, String e) {
        if (e.indexOf(".") > 0) {
            for (Method m : clazz.getMethods()) {
                if (m.getReturnType().isEnum() && m.getParameterCount() == 0) {
                    Class<?> c = m.getReturnType();
                    if (c.getSimpleName().equalsIgnoreCase(e.substring(0, e.indexOf(".")))) {
                        for (Enum en : (Enum[]) c.getEnumConstants()) {
                            if (en.name().equalsIgnoreCase(e.substring(e.indexOf(".") + 1))) {
                                return en;
                            }
                        }
                    }
                }
            }
        }
        return null;
    }

    /**
     * return "enum simple class"."enum name"
     *
     * @param e
     * @return
     */
    public static String toString(Enum e) {
        return e.getClass().getSimpleName() + '.' + e.name();
    }

    /**
     * return "[" + {@link #toString(java.lang.Enum) } for each enum + "]"
     *
     * @param l
     * @return
     */
    public static String toString(List<Enum> l) {
        StringBuilder sb = new StringBuilder().append('[');
        AtomicBoolean first = new AtomicBoolean(true);
        l.forEach((e) -> {
            if (first.get()) {
                first.set(false);
                sb.append(toString(e));
            } else {
                sb.append(", ").append(e.getClass().getSimpleName()).append('.').append(e.name());
            }

        });
        return sb.append(']').toString();
    }

    /**
     * Calls {@link #split(java.lang.String, int, boolean) } with false.
     *
     * @param toSplit
     * @param index
     * @return
     */
    public static String split(String toSplit, int index) {
        return split(toSplit, index, false);
    }

    /**
     * Call {@link #split(java.lang.String, java.lang.String, int, boolean) }
     * with ": ?" as regex.
     *
     * @param toSplit
     * @param index
     * @param ignorePatternAfterIndex
     * @return
     */
    public static String split(String toSplit, int index, boolean ignorePatternAfterIndex) {
        return split(toSplit, ": ?", index, ignorePatternAfterIndex);
    }

    /**
     * split a string on regex and return the requested index, or null. Can be
     * used to find an entity using a readable user representation of an entity
     * such as "netherlands: franeker", or "netherlands: franeker: center".
     *
     * @param toSplit
     * @param index
     * @param ignorePatternAfterIndex when true, occurring ": ?" patterns after
     * index will be ignored, so the return value will be the rest of the input
     * string.
     * @return
     */
    public static String split(String toSplit, String regex, int index, boolean ignorePatternAfterIndex) {
        if (toSplit == null) {
            return toSplit;
        }
        String[] split = toSplit.split(regex, ignorePatternAfterIndex ? index + 1 : -1);
        return (index < split.length) ? split[index] : null;
    }
    
    private static final Logger LOGGER = Logger.getLogger(Util.class.getName());

    /**
     * you can set log level to fine to see the original stacktrace in the log.
     * @param t
     * @return 
     */
    public static Throwable deepestCause(Throwable t) {
            if (LOGGER.isLoggable(Level.FINE)) {
                LOGGER.log(Level.FINE, "only retaing deepest cause of stacktrace!", t);
            }
        if (t.getCause() != null) {
            return deepestCause(t.getCause());
        }
        return t;
    }

    /**
     * find a bean programmatically in CDI provider, can be used if you need one in an unmanaged
     * situation.
     *
     * @param <T>
     * @param clazz
     * @param qualifiers if needed extend {@link AnnotationLiteral} implementing your qualifier and instantiate that.
     * @return
     */
    public static <T> T getBean(Class<T> clazz, Annotation... qualifiers) {
        BeanManager beanManager = CDI.current().getBeanManager();
        Bean<T> contextual = (Bean<T>) beanManager.getBeans(clazz, qualifiers).iterator().next();
        CreationalContext<T> ctx = beanManager.createCreationalContext(contextual);
        T bean = (T) beanManager.getReference(contextual, clazz, ctx);
        return bean;
    }
    
}
