/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.fryske_akademy.jpa;

/*-
 * #%L
 * jpaservices
 * %%
 * Copyright (C) 2018 Fryske Akademy
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */
import com.vectorprint.StringConverter;
import com.vectorprint.VectorPrintRuntimeException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Holder for parameter info that can be used when {@link JpqlBuilder building}
 * a jpql query. Uses {@link Builder#Builder() } with syntax support by default.
 * Intelligence is in the Builder, not in the Param. NOTE you can set logging level to FINE for trouble shooting
 *
 * @author eduard
 */
public class Param {
    
    private static final Logger LOGGER = Logger.getLogger(Param.class.getName());

    /**
     * Call {@link #one(java.lang.String, java.lang.Object, boolean, org.fryske_akademy.services.Param.Builder.WildcardMapping)
     * } with true, {@link Builder#DEFAULT_MAPPING} and false
     *
     * @param key
     * @param value
     * @return
     */
    public static List<Param> one(String key, Object value) {
        return one(key, value, true, Builder.DEFAULT_MAPPING, false);
    }

    /**
     * Calls {@link Builder#add(java.lang.String, java.lang.String, com.vectorprint.configuration.binding.StringConverter)
     * }
     *
     * @param key
     * @param value
     * @param converter
     * @return
     */
    public static List<Param> one(String key, String value, StringConverter converter) {
        return new Builder().add(key, value, converter).build();
    }

    /**
     * Calls {@link Builder#add(java.lang.String, java.lang.Object) }
     *
     * @param key
     * @param value
     * @param syntaxSupport
     * @param wildcardMapping
     * @param caseInsensitive
     * @return
     */
    public static List<Param> one(String key, Object value, boolean syntaxSupport, Builder.WildcardMapping wildcardMapping, boolean caseInsensitive) {
        return new Builder(syntaxSupport, wildcardMapping, caseInsensitive).add(key, value).build();
    }

    public enum AndOr {
        AND, OR;

        private static AndOr fromBool(boolean or) {
            return or ? OR : AND;
        }

        @Override
        public String toString() {
            return " " + name() + " ";
        }

    }

    private final String propertyPath;
    private final String paramKey;
    private final String operator;
    private final Object paramValue;
    private final String not;
    private final AndOr andOr;
    private final boolean caseInsensitive;

    /**
     * Bottleneck constructor
     *
     * @param propertyPath the path to the property in an entity i.e.
     * lemma.soart.id, or id
     * @param paramKey the key of the parameter i.e. :id
     * @param operator the operator i.e. like or =
     * @param paramValue may not be null
     * @param not when true use negation
     * @param or when true use or otherwise and
     * @param caseInsensitive when true query case insensitive
     */
    private Param(String propertyPath, String paramKey, String operator, Object paramValue, boolean not, boolean or, boolean caseInsensitive) {
        if (paramValue == null) {
            throw new IllegalArgumentException("param value may not be null");
        }
        this.propertyPath = propertyPath + " ";
        this.paramKey = paramKey;
        this.operator = " " + operator + " ";
        this.paramValue = paramValue;
        this.not = not ? " NOT " : " ";
        this.andOr = AndOr.fromBool(or);
        this.caseInsensitive = caseInsensitive;
    }

    public String getPropertyPath() {
        return propertyPath;
    }

    /**
     * for native queries this key should be a numeric positional parameter
     *
     * @return
     */
    public String getParamKey() {
        return paramKey;
    }

    public String getOperator() {
        return operator;
    }

    public Object getParamValue() {
        return paramValue;
    }

    public String getNot() {
        return not;
    }

    /**
     * Will be prepended in the query for parameters except the first
     *
     * @return
     */
    public AndOr getAndOr() {
        return andOr;
    }

    public Class getParamType() {
        return paramValue.getClass();
    }

    public boolean isCaseInsensitive() {
        return caseInsensitive;
    }

    @Override
    public String toString() {
        return "Param{" + "propertyPath=" + propertyPath + ", paramKey=" + paramKey + ", operator=" + operator + ", paramValue=" + paramValue + ", not=" + not + ", andOr=" + andOr + '}';
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 47 * hash + Objects.hashCode(this.paramKey);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final Param other = (Param) obj;
        if (!Objects.equals(this.paramKey, other.paramKey)) {
            return false;
        }
        return true;
    }

    /**
     * Helps building a list of Parameters, there are three approaches for
     * values.
     * <ul>
     * <li>provide string values. Syntax support: wildcards, is [not]
     * null/empty" and ! at the beginning</li>
     * <li>provide string values and a converter to get the correct object.
     * Support for ! at the beginning. {@link StringConverter} has some ready to
     * use converters</li>
     * <li>provide object values (which can be strings). No syntax support</li>
     * </ul>
     * The converter argument in methods may be null, conversion isn't applied
     * then. Methods using Object for the parameter value don't apply conversion
     * nor syntax support. When operator is not supplied, String values the
     * "like" operator, when a converter is provided "=" is the default
     * operator. Wildcard mapping, case sensitivity and syntaxSupport are
     * configured in the builder constructor.
     */
    public static class Builder {

        /**
         * translation table a wildcard for more characters and a wildcard for
         * one character
         */
        public interface WildcardMapping {

            char getMoreIn();

            char getMoreOut();

            char getOneIn();

            char getOneOut();
        }

        /**
         * * =&gt; %, ? =&gt; _, NOTE that this mapping causes all four
         * characters to be interpreted as wildcards in jpql/sql.
         */
        public static class DefaultWildcardMapping implements WildcardMapping {

            @Override
            public char getMoreIn() {
                return '*';
            }

            @Override
            public char getMoreOut() {
                return '%';
            }

            @Override
            public char getOneIn() {
                return '?';
            }

            @Override
            public char getOneOut() {
                return '_';
            }

        }

        private final List<Param> params = new ArrayList<>(3);
        public static final WildcardMapping DEFAULT_MAPPING = new DefaultWildcardMapping();
        public static final String ISNULL = "is null";
        public static final String ISNOTNULL = "is not null";
        public static final String ISEMPTY = "is empty";
        public static final String ISNOTEMPTY = "is not empty";
        public static final char NEGATION = '!';
        private final boolean syntaxInValue;
        private final WildcardMapping wildcardMapping;
        private final boolean caseInsensitive;

        public Builder(boolean syntaxInValue, WildcardMapping wildcardMapping, boolean caseInsensitive) {
            this.syntaxInValue = syntaxInValue;
            this.wildcardMapping = wildcardMapping;
            this.caseInsensitive = caseInsensitive;
        }

        /**
         * Builder with syntax support in value (!, is [not] null, is [not]
         * empty). Calls {@link #Builder(boolean, WildcardMapping, boolean)
         * } with true, null and false.
         */
        public Builder() {
            this(true, null, false);
        }

        /**
         * Calls {@link #Builder(boolean, org.fryske_akademy.services.Param.Builder.WildcardMapping, boolean)
         * } with true, null and the caseInsensitive argument.
         *
         * @param caseInsensitive
         */
        public Builder(boolean caseInsensitive) {
            this(true, null, caseInsensitive);
        }

        /**
         * This method is suitable when your paramValue is the correct object
         * for the query to be executed. It uses "like" as operator for String,
         * otherwise "=".
         *
         * @param paramKey
         * @param paramValue
         * @return
         */
        public Builder add(String paramKey, Object paramValue) {
            if (paramValue instanceof String) {
                return add(paramKey, paramKey, "like", (String) paramValue, false, null);
            } else {
                return add(paramKey, paramKey, "=", paramValue, false, true);
            }
        }

        /**
         * This method is suitable when your paramValue is the correct object
         * for the query to be executed. It uses "like" as operator for String,
         * otherwise "=".
         *
         * @param paramKey
         * @param paramValue
         * @param or will be prepended in the query for parameters except the
         * first
         * @return
         */
        public Builder add(String paramKey, Object paramValue, boolean or) {
            return add(paramKey, paramKey, paramValue instanceof String ? "like" : "=", paramValue, false, or);
        }

        /**
         * This method is suitable when your paramValue is a String that must be
         * converted to get the correct object for the query to be executed. It
         * uses "like" as operator when converter is null otherwise "=".
         *
         * @param paramKey
         * @param paramValue
         * @param converter see {@link StringConverter}
         * @return
         */
        public Builder add(String paramKey, String paramValue, StringConverter converter) {
            return add(paramKey, paramKey, converter == null ? "like" : "=", paramValue, false, converter);
        }

        /**
         * Uses true for "or"
         *
         * @param paramKey
         * @param paramValue
         * @param operator
         * @param converter
         * @return
         */
        public Builder add(String paramKey, String paramValue, String operator, StringConverter converter) {
            return add(paramKey, paramKey, operator, paramValue, true, converter);
        }

        /**
         * This method is suitable when your paramValue is a String that must be
         * converted to get the correct object for the query to be executed. It
         * uses "like" as operator when converter is null otherwise "=".
         *
         * @param paramKey
         * @param paramValue
         * @param or
         * @param converter see {@link StringConverter}
         * @return
         */
        public Builder add(String paramKey, String paramValue, boolean or, StringConverter converter) {
            return add(paramKey, paramKey, converter == null ? "=" : "like", paramValue, or, converter);
        }

        /**
         * Assumes key and propertyPath are the same
         *
         * @param paramKey
         * @param paramValue
         * @param operator
         * @param or
         * @param converter
         * @return
         */
        public Builder add(String paramKey, String paramValue, String operator, boolean or, StringConverter converter) {
            return add(paramKey, paramKey, operator, paramValue, or, converter);
        }

        /**
         * Bottleneck method for String values, when configured in the Builder,
         * applies syntax support and wildcard mapping, applies conversion of
         * the String value.
         *
         * Calls {@link #add(java.lang.String, java.lang.String, java.lang.String, java.lang.Object, boolean, boolean)
         * }
         * with {@link #operator(java.lang.String, java.lang.String, boolean) }
         * for the operator. The value will be the empty string when {@link #valueIsOperator(java.lang.String, boolean)
         * } is true, will be {@link StringConverter#convert(java.lang.String) }
         * when a converter is provided (negation is stripped), otherwise the
         * value is returned with wildcards replaced and negation stripped
         * (provided syntax support in values is active).
         *
         * @param propertyPath
         * @param paramKey
         * @param operator
         * @param paramValue
         * @param or
         * @param converter see {@link StringConverter}
         * @return
         */
        public Builder add(String propertyPath, String paramKey, String operator, String paramValue, boolean or, StringConverter converter) {
            Object value = getValue(paramValue, converter);
            if (value != null) {
                return add(propertyPath, paramKey, operator(operator, paramValue, syntaxInValue), value, isNegation(paramValue), or);
            } else {
                LOGGER.warning(String.format("skip adding param, value is null, probably %s cannot be converted by %s", paramValue, converter));
                return this;
            }
        }

        private Object getValue(String value, StringConverter converter) {
            if (valueIsOperator(value, syntaxInValue)) {
                return ""; // value doesn't matter here, not used
            } else if (converter == null) {
                return replaceWildcards(stripNegation(value));
            } else {
                try {
                    return converter.convert(stripNegation(value));
                } catch (IllegalArgumentException | VectorPrintRuntimeException e) {
                    if (LOGGER.isLoggable(Level.FINE)) {
                        LOGGER.log(Level.FINE, String.format("%s cannot be converted by %s", stripNegation(value),converter), e);
                    }
                    return null;
                }
            }
        }

        /**
         * Bottleneck method, adds a new Param, does not apply any intelligence.
         *
         * @param propertyPath
         * @param paramKey
         * @param operator
         * @param paramValue
         * @param not
         * @param or
         * @throws IllegalArgumentException when paramKey is already present or
         * when value is null
         * @return
         */
        public Builder add(String propertyPath, String paramKey, String operator, Object paramValue, boolean not, boolean or) {
            if (params.stream().anyMatch((p) -> {
                return p.paramKey.equals(paramKey);
            })) {
                throw new IllegalArgumentException(String.format("builder already contains %s", paramKey));
            }
            params.add(new Param(propertyPath, paramKey, operator, paramValue, not, or, caseInsensitive));
            return this;
        }

        public Builder remove(String paramKey) {
            if (params.stream().anyMatch((p) -> {
                return p.paramKey.equals(paramKey);
            })) {
                for (Iterator<Param> iterator = params.iterator(); iterator.hasNext();) {
                    Param next = iterator.next();
                    if (next.paramKey.equals(paramKey)) {
                        iterator.remove();
                        break;
                    }
                }
            }
            return this;
        }

        /**
         * check if a string (user value) indicates a negation and syntaxInValue
         * is used
         *
         * @see #NEGATION
         * @param value
         * @return
         */
        public boolean isNegation(String value) {
            return syntaxInValue && value != null && value.indexOf(NEGATION) == 0;
        }

        private String stripNegation(String value) {
            return isNegation(value) ? value.substring(1) : value;
        }

        private String replaceWildcards(String value) {
            return !syntaxInValue || wildcardMapping == null ? value : value
                    .replace(wildcardMapping.getMoreIn(), wildcardMapping.getMoreOut())
                    .replace(wildcardMapping.getOneIn(), wildcardMapping.getOneOut());
        }

        /**
         * Returns static final Strings when syntax is used and present in
         * value, otherwise the trimmed operator
         *
         * @param operator
         * @param value
         * @param syntaxInValue
         * @return
         */
        public static String operator(String operator, String value, boolean syntaxInValue) {
            if (syntaxInValue) {
                switch (value.trim().toLowerCase()) {
                    case ISNULL:
                        return ISNULL;
                    case ISNOTNULL:
                        return ISNOTNULL;
                    case ISEMPTY:
                        return ISEMPTY;
                    case ISNOTEMPTY:
                        return ISNOTEMPTY;
                    default:
                        return operator.trim();
                }
            } else {
                return operator.trim();
            }
        }

        /**
         * users may input "is (not) null", "is (not) empty" etc., in that case
         * there is no parameter value to be set for a key.
         *
         * @param s
         * @param syntaxInValue
         * @return
         */
        public static boolean valueIsOperator(String s, boolean syntaxInValue) {
            return syntaxInValue && (nullComp(s) || emptyComp(s));
        }

        /**
         * check if a string (user value) is a null comparison
         *
         * @see #valueIsOperator(java.lang.String, boolean)
         * @param s
         * @return
         */
        private static boolean nullComp(String s) {
            if (s == null) {
                return false;
            }
            String t = s.trim().toLowerCase();
            return t.equals(ISNULL) || t.equals(ISNOTNULL);
        }

        /**
         * check if a string (user value) is a empty comparison
         *
         * @see #valueIsOperator(java.lang.String, boolean)
         * @param s
         * @return
         */
        private static boolean emptyComp(String s) {
            if (s == null) {
                return false;
            }
            String t = s.trim().toLowerCase();
            return t.equals(ISEMPTY) || t.equals(ISNOTEMPTY);
        }

        public List<Param> build() {
            return params;
        }

        /**
         * when true wildcards are not replaced and syntax in value is not applied
         * @return 
         */
        public boolean isSyntaxInValue() {
            return syntaxInValue;
        }

        public boolean containsKey(String key) {
            return params.stream().anyMatch((t) -> {
                return t.getParamKey().equals(key);
            });
        }

    }

}
