/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.fryske_akademy.jpa;

/*-
 * #%L
 * jpaservices
 * %%
 * Copyright (C) 2018 Fryske Akademy
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

import com.vectorprint.StringConverter;
import com.vectorprint.VectorPrintRuntimeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;

/**
 * Holder for parameter info that can be used when {@link JpqlBuilder building}
 * a jpql query. Uses {@link Builder#Builder() } with syntax support by default.
 * Intelligence is in the Builder, not in the Param. NOTE you can set logging
 * level to FINE for trouble shooting
 *
 * @author eduard
 */
public class Param {

    public enum OPERATOR {
        EQ("="), GT(">"), LT("<"),GE(">="),LE("<="),
        IN("IN"),LIKE("LIKE"), NE("<>"), MEMBEROF("MEMBER OF"),
        ISNULL("IS NULL"), ISNOTNULL("IS NOT NULL"), ISEMPTY("IS EMPTY"), ISNOTEMPTY("IS NOT EMPTY");
        private OPERATOR(String token) {
            this.token=token;
        }

         private String token;

        @Override
        public String toString() {
            return " " + token + " ";
        }

        public String getToken() {
            return token;
        }

        public static OPERATOR fromToken(String token) {
            for (OPERATOR t : OPERATOR.values()) {
                if (t.token.equals(token.trim().toUpperCase())) {
                    return t;
                }
            }
            throw new IllegalArgumentException(String.format("%s invalid, only %s supported",token, Arrays.asList(OPERATOR.values())));
        }
    }

    private static final Logger LOGGER = LoggerFactory.getLogger(Param.class.getName());

    /**
     * Call
     * {@link #one(java.lang.String, java.lang.String, java.lang.Object, boolean, org.fryske_akademy.jpa.Param.Builder.WildcardMapping, boolean) }
     * with true, {@link Builder#DEFAULT_MAPPING} and false
     *
     * @param key
     * @param value
     * @return
     */
    public static List<Param> one(String key, Object value) {
        return one(key, key, value, true, Builder.DEFAULT_MAPPING, false);
    }

    /**
     * Calls {@link Builder#add(String, String, OPERATOR, Object, boolean, boolean)} with
     * key,key,operator,value,false,false
     * @param key
     * @param operator
     * @param value
     * @return
     */
    public static List<Param> one(String key, OPERATOR operator, Object value) {
        return new Builder().add(key,key,operator,value,false,false).build();
    }
    /**
     * Call
     * {@link #one(java.lang.String, java.lang.String, java.lang.Object, boolean, org.fryske_akademy.jpa.Param.Builder.WildcardMapping, boolean) }
     * with true, {@link Builder#DEFAULT_MAPPING} and false
     *
     * @param propertyPath The propertyPath may differ from key allowing you to
     * apply multiple comparisons for the same propertyPath (i.e.
     * e.column1=:column1 or e.column1=:column2).
     * @param key
     * @param value
     * @return
     */
    public static List<Param> one(String propertyPath, String key, Object value) {
        return one(propertyPath, key, value, true, Builder.DEFAULT_MAPPING, false);
    }

    /**
     * Calls {@link Builder#add(java.lang.String, java.lang.String, com.vectorprint.StringConverter) }
     * }
     *
     * @param key
     * @param value
     * @param converter
     * @return
     */
    public static List<Param> one(String key, String value, StringConverter converter) {
        return new Builder().add(key, value, converter).build();
    }

    /**
     * Calls {@link Builder#add(java.lang.String, java.lang.String, java.lang.Object, boolean)
     * } with false
     *
     * @param propertyPath The propertyPath may differ from key allowing you to
     * apply multiple comparisons for the same propertyPath (i.e.
     * e.column1=:column1 or e.column1=:column2).
     * @param key
     * @param value
     * @param syntaxSupport
     * @param wildcardMapping
     * @param caseInsensitive
     * @return
     */
    public static List<Param> one(String propertyPath, String key, Object value, boolean syntaxSupport, Builder.WildcardMapping wildcardMapping, boolean caseInsensitive) {
        return new Builder(syntaxSupport, wildcardMapping, caseInsensitive).add(propertyPath, key, value, false).build();
    }


    private final String propertyPath;
    private final String paramKey;
    private final OPERATOR operator;
    private final Object paramValue;
    private final String not;
    private final AndOr andOr;
    private final boolean caseInsensitive;
    private boolean startGroup;
    private boolean endGroup;

    /**
     * Bottleneck constructor
     *
     * @param propertyPath The propertyPath may differ from key allowing you to
     * apply multiple comparisons for the same propertyPath (i.e.
     * e.column1=:column1 or e.column1=:column2).
     * @param paramKey the key of the parameter i.e. :id
     * @param operator the operator i.e. like or =
     * @param paramValue may not be null
     * @param not when true use negation
     * @param or when true use or otherwise and
     * @param caseInsensitive when true query case insensitive
     */
    private Param(String propertyPath, String paramKey, OPERATOR operator, Object paramValue, boolean not, boolean or, boolean caseInsensitive) {
        if (paramValue == null) {
            throw new IllegalArgumentException(paramKey + ": param value may not be null");
        }
        this.propertyPath = propertyPath + " ";
        this.paramKey = paramKey;
        this.operator = operator;
        this.paramValue = paramValue;
        this.not = not ? " NOT " : " ";
        this.andOr = AndOr.fromBool(or);
        this.caseInsensitive = caseInsensitive;
    }

    public String getPropertyPath() {
        return propertyPath;
    }

    /**
     * for native queries this key should be a numeric positional parameter
     *
     * @return
     */
    public String getParamKey() {
        return paramKey;
    }

    public OPERATOR getOperator() {
        return operator;
    }

    public Object getParamValue() {
        return paramValue;
    }

    public String getNot() {
        return not;
    }

    /**
     * Will be prepended in the query for parameters except the first
     *
     * @return
     */
    public AndOr getAndOr() {
        return andOr;
    }

    public Class getParamType() {
        return paramValue.getClass();
    }

    public boolean isCaseInsensitive() {
        return caseInsensitive;
    }

    public boolean isStartGroup() {
        return startGroup;
    }

    public boolean isEndGroup() {
        return endGroup;
    }

    private void setStartGroup(boolean startGroup) {
        this.startGroup = startGroup;
    }

    private void setEndGroup(boolean endGroup) {
        this.endGroup = endGroup;
    }

    @Override
    public String toString() {
        return "Param{" +
                "propertyPath='" + propertyPath + '\'' +
                ", paramKey='" + paramKey + '\'' +
                ", operator='" + operator + '\'' +
                ", paramValue=" + paramValue +
                ", not='" + not + '\'' +
                ", andOr=" + andOr +
                ", caseInsensitive=" + caseInsensitive +
                ", startGroup=" + startGroup +
                ", endGroup=" + endGroup +
                '}';
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 47 * hash + Objects.hashCode(this.paramKey);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final Param other = (Param) obj;
        return Objects.equals(this.paramKey, other.paramKey);
    }

    /**
     * A factory for Param objects, parameter values can be added in two ways:
     * <ul>
     *     <li>as (the correct) Object => only syntax and wildcard support for String values</li>
     *     <li>as a String using a converter => syntax support also for non String values, no wildcard support</li>
     * </ul>
     */
    public static class Builder {

        public static final WildcardMapping DEFAULT_MAPPING = new DefaultWildcardMapping();
        public static final char NEGATION = '!';
        public static final String ISNULL = "is null";
        public static final String ISNOTNULL = NEGATION+"is null";
        public static final String ISEMPTY = "is empty";
        public static final String ISNOTEMPTY = NEGATION+"is empty";
        public static final char GREATER = '>';
        public static final char SMALLER = '<';
        /**
         * users may input "(!)is null", "(!)is empty", in that case there is no
         * parameter value to be set for a key.
         *
         * @param s
         * @param syntaxInValue
         * @return
         */
        public static boolean valueIsOperator(String s, boolean syntaxInValue) {
            return syntaxInValue && (nullComp(s) || emptyComp(s));
        }
        /**
         * check if a string (user value) is a null comparison
         *
         * @see #valueIsOperator(java.lang.String, boolean)
         * @param s
         * @return
         */
        private static boolean nullComp(String s) {
            if (s == null||s.isEmpty()) {
                return false;
            }
            String t = s.trim().toLowerCase();
            return t.equals(ISNULL) || t.equals(ISNOTNULL);
        }
        /**
         * check if a string (user value) is a empty comparison
         *
         * @see #valueIsOperator(java.lang.String, boolean)
         * @param s
         * @return
         */
        private static boolean emptyComp(String s) {
            if (s == null||s.isEmpty()) {
                return false;
            }
            String t = s.trim().toLowerCase();
            return t.equals(ISEMPTY) || t.equals(ISNOTEMPTY);
        }
        private final List<Param> params = new ArrayList<>(3);
        private final boolean syntaxInValue;
        private final WildcardMapping wildcardMapping;
        private final boolean caseInsensitive;

        public Builder(boolean syntaxInValue, WildcardMapping wildcardMapping, boolean caseInsensitive) {
            this.syntaxInValue = syntaxInValue;
            this.wildcardMapping = wildcardMapping;
            this.caseInsensitive = caseInsensitive;
        }

        /**
         * Builder with syntax support in value (!, >, &lt; is [not] null, is [not]
         * empty). Calls {@link #Builder(boolean, WildcardMapping, boolean)
         * } with true, null and false.
         */
        public Builder() {
            this(true, null, false);
        }

        /**
         * Calls {@link #Builder(boolean, WildcardMapping, boolean)}
         * } with true, null and the caseInsensitive argument.
         *
         * @param caseInsensitive
         */
        public Builder(boolean caseInsensitive) {
            this(true, null, caseInsensitive);
        }

        /**
         * This method is suitable when your paramValue is the correct object
         * for the query to be executed. It uses "like" as operator for String,
         * otherwise "=".
         *
         * @param paramKey
         * @param paramValue
         * @return
         */
        public Builder add(String paramKey, Object paramValue) {
            return add(paramKey, paramValue, false);
        }

        /**
         * This method is suitable when your paramValue is the correct object
         * for the query to be executed. It uses "like" as operator for String,
         * otherwise "=".
         *
         * @param paramKey
         * @param paramValue
         * @param or will be prepended to parameters in the query except the
         * first
         * @return
         */
        public Builder add(String paramKey, Object paramValue, boolean or) {
            return add(paramKey, paramKey, paramValue, or);
        }

        /**
         * This method is suitable when your paramValue is the correct object
         * for the query to be executed. The propertyPath may differ from key
         * allowing you to apply multiple comparisons for the same propertyPath
         * (i.e. e.column1=:column1 or e.column1=:column2). It uses "like" as
         * operator for String, otherwise "=".
         *
         * @param propertyPath
         * @param paramKey
         * @param paramValue
         * @param or will be prepended to parameters in the query except the
         * first
         * @return
         */
        public Builder add(String propertyPath, String paramKey, Object paramValue, boolean or) {
            if (paramValue instanceof String) {
                return add(propertyPath, paramKey, "like", (String) paramValue, or, null);
            } else {
                return add(propertyPath, paramKey, OPERATOR.EQ, paramValue, false, or);
            }
        }

        /**
         * This method is suitable when your paramValue is a String that must be
         * converted to get the correct object for the query to be executed. It
         * uses "like" as operator when converter is null otherwise "=".
         *
         * @param paramKey
         * @param paramValue
         * @param converter see {@link StringConverter}
         * @return
         */
        public Builder add(String paramKey, String paramValue, StringConverter converter) {
            return add(paramKey, paramKey, paramValue, converter);
        }

        /**
         * This method is suitable when your paramValue is a String that must be
         * converted to get the correct object for the query to be executed. The
         * propertyPath may differ from key allowing you to apply multiple
         * comparisons for the same propertyPath (i.e. e.column1=:column1 or
         * e.column1=:column2). It uses "like" as operator when converter is
         * null otherwise "=".
         *
         * @param propertyPath
         * @param paramKey
         * @param paramValue
         * @param converter see {@link StringConverter}
         * @return
         */
        public Builder add(String propertyPath, String paramKey, String paramValue, StringConverter converter) {
            return add(propertyPath, paramKey, converter == null ? "like" : "=", paramValue, false, converter);
        }

        /**
         * This method is suitable when your paramValue is a String that must be
         * converted to get the correct object for the query to be executed. It
         * uses "like" as operator when converter is null otherwise "=".
         *
         * @param paramKey
         * @param paramValue
         * @param or
         * @param converter see {@link StringConverter}
         * @return
         */
        public Builder add(String paramKey, String paramValue, boolean or, StringConverter converter) {
            return add(paramKey, paramKey, converter == null ? "like" : "=", paramValue, or, converter);
        }

        /**
         * Assumes key and propertyPath are the same
         *
         * @param paramKey
         * @param paramValue
         * @param operator
         * @param or
         * @param converter
         * @return
         */
        public Builder add(String paramKey, String paramValue, String operator, boolean or, StringConverter converter) {
            return add(paramKey, paramKey, operator, paramValue, or, converter);
        }

        /**
         * Bottleneck method for String values, when configured in the Builder,
         * applies syntax support and wildcard mapping, applies conversion of
         * the String value using the supplied converter.
         *
         * Calls {@link #add(String, String, OPERATOR, Object, boolean, boolean)}
         * }
         * with {@link #operator(java.lang.String, java.lang.String, boolean) }
         * for the operator. The value will be the empty string when {@link #valueIsOperator(java.lang.String, boolean)
         * } is true, will be {@link StringConverter#convert(java.lang.String) }
         * when a converter is provided (negation is stripped), otherwise the
         * value is returned with wildcards replaced and negation stripped
         * (provided syntax support in values is active).
         *
         * @param propertyPath The propertyPath may differ from key allowing you
         * to apply multiple comparisons for the same propertyPath (i.e.
         * e.column1=:column1 or e.column1=:column2).
         * @param paramKey
         * @param operator
         * @param paramValue
         * @param or
         * @param converter see {@link StringConverter}
         * @return
         */
        public Builder add(String propertyPath, String paramKey, String operator, String paramValue, boolean or, StringConverter converter) {
            Object value = getValue(paramValue, converter);
            if (value != null) {
                return add(propertyPath, paramKey, operator(operator, paramValue, syntaxInValue), value,
                        !valueIsOperator(paramValue, syntaxInValue) && isNegation(paramValue), or);
            } else {
                LOGGER.warn(String.format("skip adding param, value is null, probably %s cannot be converted by %s", paramValue, converter));
                return this;
            }
        }

        public Builder checkNotNull(String propertyPath) {
            return add(propertyPath, propertyPath, OPERATOR.ISNOTNULL, "", false, false);
        }

        /**
         *
         * @param group when true the last parameter starts a group (.
         * @return
         */
        public Builder lastParamStartsGroup(boolean group) {
            if (!params.isEmpty()) {
                params.get(params.size()-1).setStartGroup(group);
            }
            return this;
        }

        /**
         * @param group when true the last parameter starts a group (.
         * @return
         */
        public Builder lastParamEndsGroup(boolean group) {
            if (!params.isEmpty()) {
                params.get(params.size()-1).setEndGroup(group);
            }
            return this;
        }

        private Object getValue(String value, StringConverter converter) {
            if (valueIsOperator(value, syntaxInValue)) {
                return ""; // value doesn't matter here, not used
            } else if (converter == null) {
                return replaceWildcards(stripSyntax(value));
            } else {
                try {
                    return converter.convert(stripSyntax(value));
                } catch (IllegalArgumentException | VectorPrintRuntimeException e) {
                    if (LOGGER.isDebugEnabled()) {
                        LOGGER.debug(String.format("%s cannot be converted by %s", stripSyntax(value), converter), e);
                    }
                    return null;
                }
            }
        }

        /**
         * Bottleneck method, adds a new Param, does not apply any intelligence.
         *
         * @param propertyPath The propertyPath may differ from key allowing you
         * to apply multiple comparisons for the same propertyPath (i.e.
         * e.column1=:column1 or e.column1=:column2).
         * @param paramKey
         * @param operator
         * @param paramValue
         * @param not
         * @param or
         * @throws IllegalArgumentException when paramKey is already present or
         * when value is null
         * @return
         */
        public Builder add(String propertyPath, String paramKey, OPERATOR operator, Object paramValue, boolean not, boolean or) {
            add(new Param(propertyPath, paramKey, operator, paramValue, not, or, caseInsensitive));
            return this;
        }

        /**
         * Calls {@link #add(String, String, OPERATOR, Object, boolean, boolean)} with {@link OPERATOR#fromToken(String)}
         * @param propertyPath
         * @param paramKey
         * @param operator
         * @param paramValue
         * @param not
         * @param or
         * @return
         */
        public Builder add(String propertyPath, String paramKey, String operator, Object paramValue, boolean not, boolean or) {
            add(new Param(propertyPath, paramKey, OPERATOR.fromToken(operator), paramValue, not, or, caseInsensitive));
            return this;
        }

        private void add(Param param) {
            if (params.stream().anyMatch(p -> p.paramKey.equals(param.paramKey))) {
                throw new IllegalArgumentException(String.format("builder already contains %s", param.paramKey));
            }
            params.add(param);
        }

        public Builder remove(String paramKey) {
            if (params.stream().anyMatch((p) -> {
                return p.paramKey.equals(paramKey);
            })) {
                for (Iterator<Param> iterator = params.iterator(); iterator.hasNext();) {
                    Param next = iterator.next();
                    if (next.paramKey.equals(paramKey)) {
                        iterator.remove();
                        break;
                    }
                }
            }
            return this;
        }

        /**
         * check if a string (user value) indicates a negation when syntaxInValue
         * is used
         *
         * @see #NEGATION
         * @param value
         * @return
         */
        public boolean isNegation(String value) {
            return syntaxInValue && value.indexOf(NEGATION) == 0;
        }

        /**
         * check if a string (user value) indicates a greater than comparison when syntaxInValue
         * is used
         *
         * @see #GREATER
         * @param value
         * @return
         */
        public boolean isGreaterThan(String value) {
            return syntaxInValue && (value.indexOf(GREATER) == 0 || isNegation(value) && value.indexOf(GREATER) == 1);
        }

        /**
         * check if a string (user value) indicates a smaller than comparison when syntaxInValue
         * is used
         *
         * @see #SMALLER
         * @param value
         * @return
         */
        public boolean isSmallerThan(String value) {
            return syntaxInValue && (value.indexOf(SMALLER) == 0 || isNegation(value) && value.indexOf(SMALLER) == 1);
        }

        private String stripSyntax(String value) {
            return isNegation(value) ?
                    isGreaterThan(value) || isSmallerThan(value) ? value.substring(2) : value.substring(1) :
                    isGreaterThan(value) || isSmallerThan(value) ? value.substring(1) : value;
        }

        private String replaceWildcards(String value) {
            return wildcardMapping == null ? value : value
                    .replace(wildcardMapping.getMoreIn(), wildcardMapping.getMoreOut())
                    .replace(wildcardMapping.getOneIn(), wildcardMapping.getOneOut());
        }

        /**
         * Returns an operator from the value when syntax is used and the
         * trimmed value equals one of the supported operators, otherwise the
         * trimmed operator
         *
         * @param operator
         * @param value
         * @param syntaxInValue
         * @return
         */
        public OPERATOR operator(String operator, String value, boolean syntaxInValue) {
            if (syntaxInValue) {
                switch (value.trim().toLowerCase()) {
                    case ISNULL:
                    case ISEMPTY:
                        return OPERATOR.fromToken(value);
                    case ISNOTNULL:
                        return OPERATOR.ISNOTNULL;
                    case ISNOTEMPTY:
                        return OPERATOR.ISNOTEMPTY;
                    default:
                        if (isGreaterThan(value)) {
                            return OPERATOR.GT;
                        } else if (isSmallerThan(value)) {
                            return OPERATOR.LT;
                        } else {
                            return OPERATOR.fromToken(operator);
                        }
                }
            } else {
                return OPERATOR.fromToken(operator);
            }
        }

        public void addParam(List<Param> p) {
            p.forEach(param -> add(param));
        }

        /**
         * usefull if for example you need to add parameters yielded by a diffently configured builder
         * @return
         */
        public List<Param> build() {
            return params;
        }

        /**
         * when true wildcards are not replaced and syntax in value is not
         * applied
         *
         * @return
         */
        public boolean isSyntaxInValue() {
            return syntaxInValue;
        }

        public boolean containsKey(String key) {
            return params.stream().anyMatch((t) -> {
                return t.getParamKey().equals(key);
            });
        }
        /**
         * * =&gt; %, ? =&gt; _, NOTE that this mapping causes all four
         * characters to be interpreted as wildcards in jpql/sql.
         */
        public static class DefaultWildcardMapping implements WildcardMapping {
            
            @Override
            public char getMoreIn() {
                return '*';
            }
            
            @Override
            public char getMoreOut() {
                return '%';
            }
            
            @Override
            public char getOneIn() {
                return '?';
            }
            
            @Override
            public char getOneOut() {
                return '_';
            }
            
        }
        /**
         * translation table a wildcard for more characters and a wildcard for
         * one character
         */
        public interface WildcardMapping {
            
            char getMoreIn();
            
            char getMoreOut();
            
            char getOneIn();
            
            char getOneOut();
        }

    }
    public enum AndOr {
        AND, OR;
        
        private static AndOr fromBool(boolean or) {
            return or ? OR : AND;
        }
        
        @Override
        public String toString() {
            return " " + name() + " ";
        }
        
    }

}
