/**
 * Copyright (C) 2015 Michael Schnell. All rights reserved.
 * http://www.fuin.org/
 * <p>
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 * <p>
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * <p>
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see http://www.gnu.org/licenses/.
 */
package org.fuin.ddd4j.core;

import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;

import java.io.Serial;

/**
 * Base class for version related aggregate exceptions.
 */
public abstract class AbstractVersionedAggregateException extends AbstractAggregateException {

    @Serial
    private static final long serialVersionUID = 1L;

    private final int version;

    /**
     * Constructor with strongly typed data.
     *
     * @param message Error message.
     * @param type    Type of the aggregate.
     * @param id      Unique identifier of the aggregate.
     * @param version Version of the aggregate.
     */
    public AbstractVersionedAggregateException(@NotEmpty final String message,
                                               @NotNull final EntityType type,
                                               @NotNull final AggregateRootId id,
                                               final int version) {
        this(message, type.asString(), id.asString(), version);
    }

    /**
     * Constructor with string data.
     *
     * @param message Error message.
     * @param type    Type of the aggregate.
     * @param id      Unique identifier of the aggregate.
     * @param version Version of the aggregate.
     */
    public AbstractVersionedAggregateException(@NotEmpty final String message,
                                               @NotNull final String type,
                                               @NotNull final String id,
                                               final int version) {
        super(message, type, id);
        this.version = version;
    }

    /**
     * Returns the actual version.
     *
     * @return Actual version.
     */
    public final int getVersion() {
        return version;
    }

}
