package org.fulib.workflows.generators;

import org.fulib.workflows.events.*;
import org.fulib.workflows.generators.constructors.ClassDiagramConstructor;
import org.fulib.workflows.generators.constructors.ObjectDiagramConstructor;
import org.fulib.yaml.YamlObject;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

/**
 * The DiagramGenerator manages the building and generation of object and class diagrams.
 */
public class DiagramGenerator {
    private BoardGenerator boardGenerator;

    DiagramGenerator(BoardGenerator boardGenerator) {
        this.boardGenerator = boardGenerator;
    }

    /**
     * Builds and generates diagrams from an event storming Board
     *
     * @param board generated by the fulibWorkflows yaml parser
     */
    public void buildAndGenerateDiagram(Board board) {
        Map<String, String> generatedDiagrams = buildDiagrams(board);

        for (String key : generatedDiagrams.keySet()) {
            generateDiagram(generatedDiagrams.get(key), key, key.startsWith("classDiagram"));
        }
    }

    /**
     * Builds object and class structure and files from event storming Board
     *
     * @param board generated by the fulibWorkflows yaml parser
     * @return Map containing all object and classdiagrams as string value, key consists of an index and classdiagram/object
     */
    public Map<String, String> buildDiagrams(Board board) {
        ObjectDiagramConstructor diagramConstructor = new ObjectDiagramConstructor();

        Map<String, String> resultMap = new HashMap<>();

        List<String> diagrams = new ArrayList<>();

        Map<String, List<Data>> previousServiceData = new HashMap<>();
        String currentService = "default";

        List<Data> previousData = new ArrayList<>();

        // ObjectDiagrams
        for (Workflow workflow : board.getWorkflows()) {
            for (BaseNote note : workflow.getNotes()) {
                if (note instanceof Service) {
                    Service service = (Service) note;
                    currentService = service.getName();
                }
                else if (note instanceof Data) {
                    previousData = previousServiceData.computeIfAbsent(currentService, k -> new ArrayList<>());
                    previousData.add((Data) note);

                    // Always use current note and all previous to represent the objectDiagram according to the timeline
                    String diagramString = diagramConstructor.buildObjectDiagram(previousData, note.getIndex());

                    if (diagramString != null) {
                        diagrams.add(diagramString);
                    }
                }
            }
        }

        for (int i = 0; i < diagrams.size(); i++) {
            String diagram = diagrams.get(i);
            resultMap.put(i + "_diagram", diagram);
        }

        ClassDiagramConstructor classDiagramConstructor = new ClassDiagramConstructor();

        // ClassDiagrams
        for (Entry<String, List<Data>> entry : previousServiceData.entrySet()) {
            String key = entry.getKey();
            previousData = entry.getValue();
            Map<String, YamlObject> yamlGraph = diagramConstructor.buildFulibGraphDiagram(previousData);
            String classdiagramString = classDiagramConstructor.buildClassDiagram(previousData, yamlGraph);

            if (classdiagramString != null) {
                resultMap.put("classDiagram_" + key, classdiagramString);
            }
        }

        return resultMap;
    }

    private void generateDiagram(String diagramContent, String fileName, boolean isClass) {
        try {
            String outputDirectory = boardGenerator.getGenDir();
            outputDirectory += isClass ? "/class/" : "/diagrams/";

            Files.createDirectories(Path.of(outputDirectory));

            String outputDiagramFilePath = outputDirectory + fileName + ".svg";
            if (!Files.exists(Path.of(outputDiagramFilePath))) {
                Files.createFile(Path.of(outputDiagramFilePath));
            }
            Files.writeString(Path.of(outputDiagramFilePath), diagramContent);

        } catch (IOException e) {
            e.printStackTrace();
        }
    }
}
