package org.fulib.workflows.generators;

import org.antlr.v4.runtime.misc.Pair;
import org.fulib.workflows.events.*;
import org.fulib.workflows.generators.constructors.BoardConstructor;
import org.fulib.workflows.generators.constructors.PageConstructor;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.*;

/**
 * The HtmlGenerator manages the building and generation of an event storming board and mockup pages described as page events.
 */
public class HtmlGenerator {
    private final Map<String, Integer> allPagesMap = new HashMap<>();
    private final Map<String, Page> divPageMap = new LinkedHashMap<>();
    private BoardGenerator boardGenerator;

    HtmlGenerator(Board board, BoardGenerator boardGenerator) {
        createAllPagesMap(board);
        this.boardGenerator = boardGenerator;
    }

    /**
     * Builds and generates an event storming board and page mockups from event storming Board
     *
     * @param board generated by the fulibWorkflows yaml parser
     */
    public void buildAndGenerateHTML(Board board) {
        Map<String, String> generatedHTMLs = buildHTMLs(board);

        for (String key : generatedHTMLs.keySet()) {
            if (key.equals("Board")) {
                generateHTML(generatedHTMLs.get(key), key, "");
            } else {
                generateHTML(generatedHTMLs.get(key), key, "pages/");
            }
        }
    }

    /**
     * Builds an event storming board and page mockups from event storming Board
     *
     * @param board generated by the fulibWorkflows yaml parser
     * @return Map containing a board and page mockups as string value, key consists of an index and Board/page
     */
    public Map<String, String> buildHTMLs(Board board) {
        BoardConstructor boardConstructor = new BoardConstructor();
        PageConstructor pageConstructor = new PageConstructor();

        Map<String, String> resultMap = new HashMap<>();

        String htmlBoard = boardConstructor.buildBoard(board, boardGenerator.isWebGeneration());
        resultMap.put("Board", htmlBoard);

        List<String> pagesHTML = new ArrayList<>();

        Page lastPage = null;
        for (Workflow workflow : board.getWorkflows()) {
            for (BaseNote note : workflow.getNotes()) {
                if (note instanceof Page page) {
                    lastPage = page;
                    List<Integer> targetPageIndexList = evaluateTargetPageIndex(page);
                    String pageContent = pageConstructor.buildPage(page, targetPageIndexList, divPageMap, this.boardGenerator.isWebGeneration());
                    if (pageContent != null) {
                        pagesHTML.add(pageContent);
                    }
                }
                if (note instanceof Div div) {
                    Page divPage = new Page();
                    divPage.setName(div.getName());
                    divPage.setIndex(div.getIndex());
                    divPage.setContent(div.getContent());
                    divPageMap.put(div.getName(), divPage);

                    if (lastPage != null) {
                        List<Integer> targetPageIndexList = evaluateTargetPageIndex(lastPage);

                        String lastPageContent = pageConstructor.buildPage(lastPage, targetPageIndexList, divPageMap, this.boardGenerator.isWebGeneration());
                        if (lastPageContent != null) {
                            pagesHTML.add(lastPageContent);
                        }
                    }
                }
            }
        }

        for (int i = 0; i < pagesHTML.size(); i++) {
            String page = pagesHTML.get(i);
            resultMap.put(i + "_page", page);
        }

        return resultMap;
    }

    private List<Integer> evaluateTargetPageIndex(Page page) {
        List<Integer> result = new ArrayList<>();

        for (int i = 0; i < page.getContent().size(); i++) {
            Pair<String, String> elementPair = page.getContent().get(i);
            if (elementPair.a.equals("targetPage")) {
                String targetPageName = elementPair.b;

                Integer targetIndex = allPagesMap.get(targetPageName);

                if (targetIndex != null) {
                    result.add(targetIndex);
                }
            }
        }

        return result;
    }

    private void generateHTML(String htmlContent, String fileName, String subFolder) {
        try {
            String outputDirectory = boardGenerator.getGenDir() + "/" + subFolder;
            Files.createDirectories(Path.of(outputDirectory));

            String outputBoardFilePath = outputDirectory + fileName + ".html";
            Path outputPath = Path.of(outputBoardFilePath);
            if (!Files.exists(outputPath)) {
                Files.createFile(outputPath);
            }
            Files.writeString(outputPath, htmlContent);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    private void createAllPagesMap(Board board) {
        int index = 0;

        for (Workflow workflow : board.getWorkflows()) {
            for (BaseNote note : workflow.getNotes()) {
                if (note instanceof Page page) {
                    allPagesMap.put(page.getName(), index);
                    index++;
                }
            }
        }
    }
}
