package org.fulib.workflows.generators.constructors;

import org.antlr.v4.runtime.misc.Pair;
import org.fulib.workflows.events.*;
import org.stringtemplate.v4.ST;
import org.stringtemplate.v4.STGroupFile;

import java.net.URL;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;

import static org.fulib.workflows.utils.Constants.*;

/**
 * The BoardConstructor builds the event storming boards from fulibWorkflows as html.
 */
public class BoardConstructor {
    private Board currentBoard;
    private STGroupFile boardGroup;
    private boolean webGeneration;
    private final Map<String, Page> divPageMap = new LinkedHashMap<>();
    private boolean pageComplete;
    private Page lastPage;

    /**
     * Builds an event storming board
     *
     * @param board generated by the fulibWorkflows yaml parser
     * @return html content for an event storming board as String
     */
    public String buildBoard(Board board, boolean webGeneration) {
        this.webGeneration = webGeneration;
        currentBoard = board;
        URL resource = PageConstructor.class.getResource("Board.stg");

        boardGroup = new STGroupFile(Objects.requireNonNull(resource));
        StringBuilder boardBody = new StringBuilder();

        String boardContent = buildBoardContent();

        ST st = boardGroup.getInstanceOf("board");
        st.add("content", boardContent);

        boardBody.append(st.render());
        return boardBody.toString();
    }

    private String buildBoardContent() {
        ST st;
        StringBuilder boardContent = new StringBuilder();

        for (Workflow workflow : currentBoard.getWorkflows()) {
            String htmlWorkflow = buildWorkflow(workflow);

            st = boardGroup.getInstanceOf("workflow");
            st.add("name", workflow.getName());
            st.add("content", htmlWorkflow);
            boardContent.append(st.render());
        }

        return boardContent.toString();
    }

    private String buildWorkflow(Workflow workflow) {
        ST noteST;
        ST actorST;
        ST objectNoteST;
        StringBuilder workflowContent = new StringBuilder();
        int pageIndex = 0;
        int dataIndex = 0;

        for (BaseNote note : workflow.getNotes()) {
            noteST = boardGroup.getInstanceOf("note");
            actorST = boardGroup.getInstanceOf("actor");
            objectNoteST = boardGroup.getInstanceOf("objectNoteAlone");
            if (this.webGeneration) {
                objectNoteST = boardGroup.getInstanceOf("objectNote");
            }

            if (note instanceof Event event) {
                noteST.add("name", "Event");
                noteST.add("content", buildNoteContentFromNote(event, "Event"));
                noteST.add("color", "orange");
                workflowContent.append(noteST.render());
            } else if (note instanceof Command) {
                noteST.add("name", "Command");
                noteST.add("content", buildNoteContent(note.getName()));
                noteST.add("color", "lightskyblue");
                workflowContent.append(noteST.render());
            } else if (note instanceof Policy) {
                noteST.add("name", "Policy");
                noteST.add("content", buildNoteContent(note.getName()));
                noteST.add("color", "#C8A2C8");
                workflowContent.append(noteST.render());
            } else if (note instanceof Problem) {
                noteST.add("name", "Problem");
                noteST.add("content", buildNoteContent(note.getName()));
                noteST.add("color", "indianred");
                workflowContent.append(noteST.render());
            } else if (note instanceof User) {
                actorST.add("color", "gold");
                actorST.add("icon", "person-fill");
                actorST.add("name", note.getName());
                workflowContent.append(actorST.render());
            } else if (note instanceof ExternalSystem) {
                actorST.add("color", "pink");
                actorST.add("icon", "hdd-network");
                actorST.add("name", note.getName());
                workflowContent.append(actorST.render());
            } else if (note instanceof Service) {
                actorST.add("color", "palevioletred");
                actorST.add("icon", "pc-horizontal");
                actorST.add("name", note.getName());
                workflowContent.append(actorST.render());
            } else if (note instanceof Data data) {
                objectNoteST.add("content", buildNoteContentFromNote(data, "Data"));
                workflowContent.append(objectNoteST.render());
                closeDataOrPageNote(workflowContent, dataIndex, false);
                dataIndex++;
            } else if (note instanceof Page page) {
                lastPage = page;
                pageComplete = true;
                String pageNote = buildPageNote(page);
                workflowContent.append(pageNote);
                closeDataOrPageNote(workflowContent, pageIndex, true);
                if (pageComplete) {
                    pageIndex++;
                }
            } else if (note instanceof Div div) {
                pageComplete = true;
                Page divPage = new Page();
                divPage.setName(div.getName());
                divPage.setIndex(div.getIndex());
                divPage.setContent(div.getContent());
                divPageMap.put(div.getName(), divPage);

                String pageNote = buildPageNote(divPage);
                workflowContent.append(pageNote);
                closeDataOrPageNote(workflowContent, pageIndex, true);

                if (lastPage != null) {
                    buildPageNote(lastPage);
                }

                if (pageComplete) {
                    pageIndex++;
                }
            } else if (note instanceof Develop develop) {
                String content = buildDevelop(develop);
                workflowContent.append(content);
            }
        }

        return workflowContent.toString();
    }

    private String buildDevelop(Develop develop) {
        // just add all key value pairs
        StringBuilder result = new StringBuilder();
        for (Map.Entry<Integer, Pair<String, String>> entry : develop.getContent().entrySet()) {
            Pair<String, String> pair = entry.getValue();
            String value = pair.b.replaceAll("\n", "<br>\n");
            String line = String.format("%s: %s<br>\n", pair.a, value);
            result.append(line);
        }

        // develop(content)
        ST st = boardGroup.getInstanceOf("develop");
        st.add("content", result.toString());
        return st.render();
    }

    private void closeDataOrPageNote(StringBuilder workflowContent, int index, boolean isPage) {
        StringBuilder closeContent = new StringBuilder();
        ST st;

        if (webGeneration) {
            st = boardGroup.getInstanceOf("linkedNoteLink");
            st.add("index", index);

            if (isPage) {
                st.add("type", PAGE_TYPE);
                st.add("description", PAGE_DESCRIPTION);
            } else {
                st.add("type", OBJECT_DIAGRAM_TYPE);
                st.add("description", OBJECTDIAGRAM_DESCRIPTION);
            }
            closeContent.append(st.render());
        }

        st = boardGroup.getInstanceOf("dataANDPageClose");
        closeContent.append(st.render());

        workflowContent.append(closeContent);
    }

    private String buildNoteContent(String content) {
        ST st;

        StringBuilder textContent = new StringBuilder();

        st = boardGroup.getInstanceOf("cardText");
        st.add("text", content);
        textContent.append(st.render());

        return textContent.toString();
    }

    private String buildNoteContentFromNote(BaseNote note, String noteType) {
        Map<Integer, Pair<String, String>> contents = new HashMap<>();

        ST st;

        StringBuilder textContent = new StringBuilder();

        switch (noteType) {
            case "Event" -> {
                contents = ((Event) note).getData();
                st = boardGroup.getInstanceOf("cardText");
                st.add("text", "name: " + note.getName());
                textContent.append(st.render());
            }
            case "Data" -> {
                contents = ((Data) note).getData();
                st = boardGroup.getInstanceOf("cardText");
                st.add("text", "name: " + note.getName());
                textContent.append(st.render());
            }
            default -> System.out.println("Unknown Type");
        }

        for (int i = 0; i <= contents.size(); i++) {
            Pair<String, String> pair = contents.get(i);

            if (pair == null) {
                continue;
            }

            String text = pair.a + ": ";
            text += pair.b;

            st = boardGroup.getInstanceOf("cardText");
            st.add("text", text);
            textContent.append(st.render());
        }

        return textContent.toString();
    }

    private String buildPageNote(Page page) {
        ST pageNoteST = boardGroup.getInstanceOf("pageNote");

        String pageContent = buildPageContent(page);

        pageNoteST.add("name", page.getName());
        pageNoteST.add("content", pageContent);

        return pageNoteST.render();
    }

    private String buildPageContent(Page page) {
        StringBuilder pageContent = new StringBuilder();
        Map<Integer, Pair<String, String>> contents = page.getContent();

        for (int i = 0; i < contents.size(); i++) {
            ST st;
            Pair<String, String> pair = contents.get(i);

            String desc = pair.a;
            String value = pair.b;

            switch (desc) {
                case "text" -> {
                    st = boardGroup.getInstanceOf("pageText");
                    if (value.contains("<pre>")) {
                        value = value.replace("<pre>", "<pre  style=\"text-align:left\">");
                        st = boardGroup.getInstanceOf("pagePre");
                    }
                    st.add("text", value);
                    pageContent.append(st.render());
                }
                case "input", "password" -> {
                    st = boardGroup.getInstanceOf("pageInput");
                    st.add("desc", value);
                    pageContent.append(st.render());
                }
                case "value" -> {
                    st = boardGroup.getInstanceOf("pageValue");
                    st.add("value", value);
                    pageContent.append(st.render());
                }
                case "button" -> {
                    st = boardGroup.getInstanceOf("pageButton");
                    st.add("desc", value);
                    pageContent.append(st.render());
                }
                case "div" -> {

                    st = boardGroup.getInstanceOf("pageText");
                    st.add("text", value);
                    pageContent.append(st.render());

                    // is the page complete?
                    String[] split = stripBraces(value).split(",");
                    for (int j = 0; j < split.length; j++) {
                        split[j] = split[j].trim();
                        Page divPage = divPageMap.get(split[j]);
                        if (divPage != null) {
                            String content = buildPageNote(divPage);
                        } else {
                            pageComplete = false;
                        }
                    }
                }
                default -> {
                }
            }
        }

        return pageContent.toString();
    }

    private String stripBraces(String value) {
        int pos = value.indexOf('[');
        if (pos >= 0) {
            value = value.substring(pos + 1);
        }
        pos = value.indexOf(']');
        if (pos >= 0) {
            value = value.substring(0, pos);
        }
        return value;
    }
}
