/**
 * Copyright (c) 2012 - 2019 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - Initial implementation 
 */
package org.gecko.eclipse.compatibility;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Logger;

import org.eclipse.core.runtime.internal.adaptor.EclipseAppLauncher;
import org.eclipse.osgi.framework.log.FrameworkLog;
import org.eclipse.osgi.internal.framework.EquinoxConfiguration;
import org.eclipse.osgi.service.environment.EnvironmentInfo;
import org.eclipse.osgi.service.runnable.ApplicationLauncher;
import org.gecko.eclipse.compatibility.equinox.config.api.EquinoxConfigInitializer;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleException;
import org.osgi.framework.Filter;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.application.ApplicationDescriptor;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ReferencePolicy;
import org.osgi.util.promise.PromiseFactory;
import org.osgi.util.tracker.ServiceTracker;
import org.osgi.util.tracker.ServiceTrackerCustomizer;

/**
 * Component taking the Job of the EclipsePlattformStarter that starts the configured Application as soon as the 
 * {@link ApplicationDescriptor} becomes available
 * 
 * @author Juergen Albert
 */
@Component(immediate=true)
public class EclipsePlatformStarter implements ServiceTrackerCustomizer<ApplicationDescriptor, ApplicationDescriptor> {

	private static ExecutorService executorService = Executors.newSingleThreadExecutor(new ThreadFactory() {
		
		@Override
		public Thread newThread(Runnable r) {
			return new Thread(r, "Gecko RCP Application Runner Promise");
		}
	});

	private Logger logger = Logger.getLogger(EclipsePlatformStarter.class.getName());
	
	@Reference(policy=ReferencePolicy.STATIC, cardinality=ReferenceCardinality.MANDATORY)
	EquinoxConfigInitializer marker;
	
	@Reference
	FrameworkLog log;
	
	@Reference
	EnvironmentInfo envInfo;
	

	private EclipseAppLauncher appLauncher;

	private ServiceRegistration<ApplicationLauncher> appLauncherRegistration;

	private AtomicBoolean shutdown = new AtomicBoolean(true);

	private BundleContext ctx;

	private ServiceTracker<ApplicationDescriptor, ApplicationDescriptor> descriptorTracker;

	private String applicationId;
	
	@Activate
	public void activate(BundleContext ctx) throws InvalidSyntaxException {
		this.ctx = ctx;
		Bundle runtime = findBundle("org.eclipse.core.runtime", ctx);
		if(runtime != null) {
			try {
				runtime.start();
			} catch (BundleException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}
		
		String ignoreApp = ctx.getProperty("eclipse.ignoreApp");
		if(!Boolean.parseBoolean(ignoreApp)) {
			applicationId = ctx.getProperty("eclipse.application");
			if(applicationId == null) {
				logger.severe("Can't start Equinox Application because no eclipse.application property is defined");
				return;
			}
			Filter filter = FrameworkUtil.createFilter(String.format("(&(objectClass=%s)(service.pid=%s))", ApplicationDescriptor.class.getName(), applicationId));
			descriptorTracker = new ServiceTracker<ApplicationDescriptor, ApplicationDescriptor>(ctx, filter, this);
			descriptorTracker.open();
			System.out.println("Opended descriptor tracker for " + applicationId);
		}
	}
	
	@Deactivate
	public void deactivate() {
		System.out.println("Stopping Application " + applicationId);
		descriptorTracker.close();
		appLauncherRegistration.unregister();
		shutdown.set(false);
		appLauncher.shutdown();
		System.out.println("Stopped Application " + applicationId);
	}

	private Bundle findBundle(String bsn, BundleContext ctx) {
		Bundle[] bundles = ctx.getBundles();
		for(Bundle bundle : bundles) {
			if(bsn.equals(bundle.getSymbolicName())) {
				return bundle;
			}
		}
		return null;
	}

	@Override
	public ApplicationDescriptor addingService(ServiceReference<ApplicationDescriptor> arg0) {
		startApplication();
		return ctx.getService(arg0);
	}

	private void startApplication() {
		PromiseFactory promiseFactory = new PromiseFactory(executorService);
		
		logger.info("Registering Equinox App  Launcher");
		
		// create the ApplicationLauncher and register it as a service
		appLauncher = new EclipseAppLauncher(ctx, false, false, log, (EquinoxConfiguration) envInfo);
		logger.info("Starting Equinox App Launcher");
		
		// must start the launcher AFTER service registration because this method 
		// blocks and runs the application on the current thread.  This method 
		promiseFactory.submit(() -> {
			// will return only after the application has stopped.
			System.out.println("calling start");
			Object start = appLauncher.start(null);	
			return start;
		}).thenAccept(o -> {
			if(shutdown.get()) {
				logger.info("Shutting down with Application " +  applicationId + " return code: " + o);
				Bundle bundle = ctx.getBundle(0);
				logger.info("Stopping Framework");
				bundle.stop();
				logger.info("Stopped Framework");
			} 
		}).onFailure(t -> {
			t.printStackTrace();
			System.exit(42);	
		});
		appLauncherRegistration = ctx.registerService(ApplicationLauncher.class, appLauncher, null);
	}

	@Override
	public void modifiedService(ServiceReference<ApplicationDescriptor> arg0, ApplicationDescriptor arg1) {
	}


	@Override
	public void removedService(ServiceReference<ApplicationDescriptor> arg0, ApplicationDescriptor arg1) {
		ctx.ungetService(arg0);
	}
}
