/*
 * Copyright 2017 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.filerepository.service.ftp;

import java.util.Collections;
import java.util.List;

import org.apache.ftpserver.ftplet.Authority;
import org.apache.ftpserver.ftplet.AuthorizationRequest;
import org.apache.ftpserver.ftplet.User;

/**
 * The Class FtpUser.
 */
public class FtpUser implements User {

	/** The name. */
	private String name;

	/** The password. */
	private String password;

	/** The authorities. */
	private List<? extends Authority> authorities;

	/** The max idle time. */
	private int maxIdleTime;

	/** The enabled. */
	private boolean enabled;

	/** The home directory. */
	private String homeDirectory;

	/**
	 * Instantiates a new ftp user.
	 */
	public FtpUser() {
	}

	/**
	 * Instantiates a new ftp user.
	 *
	 * @param username the username
	 */
	public FtpUser(final String username) {
		name = username;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.User#getName()
	 */
	@Override
	public String getName() {
		return name;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.User#getPassword()
	 */
	@Override
	public String getPassword() {
		return password;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.User#getAuthorities()
	 */
	@Override
	public List<? extends Authority> getAuthorities() {
		return Collections.unmodifiableList(authorities);
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.User#getAuthorities(java.lang.Class)
	 */
	@Override
	public List<? extends Authority> getAuthorities(final Class<? extends Authority> clazz) {
		return null;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.User#authorize(org.apache.ftpserver.ftplet.
	 * AuthorizationRequest)
	 */
	@Override
	public AuthorizationRequest authorize(AuthorizationRequest request) {
		// check for no authorities at all
		if (authorities == null) {
			return null;
		}

		boolean someoneCouldAuthorize = false;
		for (final Authority authority : authorities) {
			if (authority.canAuthorize(request)) {
				someoneCouldAuthorize = true;

				request = authority.authorize(request);

				// authorization failed, return null
				if (request == null) {
					return null;
				}
			}

		}

		if (someoneCouldAuthorize) {
			return request;
		} else {
			return null;
		}
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.User#getMaxIdleTime()
	 */
	@Override
	public int getMaxIdleTime() {
		return maxIdleTime;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.User#getEnabled()
	 */
	@Override
	public boolean getEnabled() {
		return enabled;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.User#getHomeDirectory()
	 */
	@Override
	public String getHomeDirectory() {
		return homeDirectory;
	}

	/**
	 * Sets the name.
	 *
	 * @param name the name to set
	 */
	public final void setName(final String name) {
		this.name = name;
	}

	/**
	 * Sets the password.
	 *
	 * @param password the password to set
	 */
	public final void setPassword(final String password) {
		this.password = password;
	}

	/**
	 * Sets the authorities.
	 *
	 * @param authorities the authorities to set
	 */
	public final void setAuthorities(final List<? extends Authority> authorities) {
		this.authorities = authorities;
	}

	/**
	 * Sets the max idle time.
	 *
	 * @param maxIdleTime the maxIdleTime to set
	 */
	public final void setMaxIdleTime(final int maxIdleTime) {
		this.maxIdleTime = maxIdleTime;
	}

	/**
	 * Sets the enabled.
	 *
	 * @param enabled the enabled to set
	 */
	public final void setEnabled(final boolean enabled) {
		this.enabled = enabled;
	}

	/**
	 * Sets the home directory.
	 *
	 * @param homeDirectory the homeDirectory to set
	 */
	public final void setHomeDirectory(final String homeDirectory) {
		this.homeDirectory = homeDirectory;
	}

}
