/*
 * Copyright 2018 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.filerepository.service.ftp;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.ftpserver.ftplet.AuthenticationFailedException;
import org.apache.ftpserver.ftplet.FileSystemFactory;
import org.apache.ftpserver.ftplet.FileSystemView;
import org.apache.ftpserver.ftplet.FtpException;
import org.apache.ftpserver.ftplet.FtpFile;
import org.apache.ftpserver.ftplet.User;
import org.genesys.filerepository.InvalidRepositoryFileDataException;
import org.genesys.filerepository.InvalidRepositoryPathException;
import org.genesys.filerepository.NoSuchRepositoryFileException;
import org.genesys.filerepository.model.RepositoryFile;
import org.genesys.filerepository.service.RepositoryService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.AuthenticationException;
import org.springframework.stereotype.Component;

/**
 * A factory for creating RepositoryFileSystem objects.
 */
@Component
public class RepositoryFileSystemFactory implements FileSystemFactory, InitializingBean {

	/** The Constant LOG. */
	private final static Logger LOG = LoggerFactory.getLogger(RepositoryFileSystemFactory.class);

	/** The repository service. */
	@Autowired(required = true)
	private RepositoryService repositoryService;

	/** The bytes manager. */
	@Autowired(required = true)
	private TemporaryBytesManager bytesManager;


	/**
	 * File.
	 *
	 * @param repositoryFile the repository file
	 * @return the repository ftp file
	 */
	private RepositoryFtpFile file(final RepositoryFile repositoryFile, final RepositoryFileSystemView session) {
		LOG.trace("Making RepositoryFtpFile repositoryFile={}", repositoryFile);

		final RepositoryFtpFile rff = new RepositoryFtpFile(repositoryFile) {

			@Override
			public String getOwnerName() {
				// TODO Auto-generated method stub
				return "root";
			}

			@Override
			public String getGroupName() {
				// TODO Auto-generated method stub
				return "wheel";
			}

			@Override
			public boolean mkdir() {
				// TODO Auto-generated method stub
				return false;
			}

			@Override
			public boolean delete() {
				return FtpRunAs.asFtpUser(session.user, () -> {
					LOG.info("Delete file={}", this.getAbsolutePath());
					try {
						repositoryService.removeFile(repositoryFile);
						return true;
					} catch (NoSuchRepositoryFileException | IOException e) {
						LOG.warn(e.getMessage());
						return false;
					}
				});
			}

			@Override
			public boolean move(final FtpFile destination) {
				return FtpRunAs.asFtpUser(session.user, () -> {
					LOG.info("Move file={} to dest={}", this.getAbsolutePath(), destination.getAbsolutePath());
					try {
						repositoryService.moveAndRenameFile(repositoryFile, destination.getAbsolutePath());
						return true;
					} catch (InvalidRepositoryPathException | InvalidRepositoryFileDataException e) {
						LOG.warn("Error moving file: {}", e.getMessage());
						return false;
					}
				});
			}

			@Override
			public OutputStream createOutputStream(final long offset) throws IOException {
				return FtpRunAs.asFtpUser(session.user, () -> {
					LOG.info("Creating output stream for file={} at offset={}", getAbsolutePath(), offset);
					return bytesManager.createOutputStream(session.user, repositoryFile, offset);
				});
			}

			@Override
			public InputStream createInputStream(final long offset) throws IOException {
				return FtpRunAs.asFtpUser(session.user, () -> {
					LOG.info("Creating input stream for file={} at offset={}", getAbsolutePath(), offset);
					return bytesManager.createInputStream(repositoryFile, offset);
				});
			}

		};

		return rff;
	}

	/**
	 * Directory.
	 *
	 * @param path the path
	 * @param session the session
	 * @return the repository ftp directory
	 */
	private RepositoryFtpDirectory directory(final String path, final RepositoryFileSystemView session) {
		LOG.trace("Making RepositoryFtpDirectory path={}", path);
		final RepositoryFtpDirectory rfd = new RepositoryFtpDirectory(path) {

			@Override
			public boolean move(final FtpFile destination) {
				return FtpRunAs.asFtpUser(session.user, () -> {
					LOG.info("Move directory={} to dest={}", this.getAbsolutePath(), destination.getAbsolutePath());
					try {
						repositoryService.renamePath(this.getAbsolutePath(), destination.getAbsolutePath());
						return true;
					} catch (final InvalidRepositoryPathException e) {
						LOG.error("Failed to rename directory", e);
						return false;
					}
				});
			}

			@Override
			public boolean mkdir() {
				LOG.info("Mkdir directory={}", this.getAbsolutePath());
				// TODO Auto-generated method stub
				return false;
			}

			@Override
			public List<? extends FtpFile> listFiles() {
				return FtpRunAs.asFtpUser(session.user, () -> {
					return _listFiles();
				});
			}
			
			private List<? extends FtpFile> _listFiles() {
				final String currentPath = getAbsolutePath();
				LOG.debug("Listing files in path={}", currentPath);
				final ArrayList<FtpFile> all = new ArrayList<>();

				final Path root = Paths.get(currentPath);

				all.addAll(repositoryService.getFiles(currentPath).stream().peek(rf -> {
					// System.err.println("repoFile " + rf.getPath() + " " +
					// rf.getOriginalFilename());
				}).map(rf -> file(rf, session)).collect(Collectors.toList()));

				try {
					all.addAll(repositoryService.listPaths(currentPath).stream()
						// remove from temporaryDir if exists in the repository
						.peek(path -> {
							session.temporaryDirs.remove(path);
						})
						// filter out current path
						.filter(path -> !currentPath.equals(path))
						// get the first subfolder name within currentPath
						.map(path -> {
							final Path foo = Paths.get(path.substring(currentPath.length()));
							LOG.trace("Sub cp=" + currentPath + " p=" + path + " x=" + foo.getName(0));
							return foo.getName(0).toString();
						})
						// remove duplicates and make directories
						.distinct().map(path -> directory(path, session)).collect(Collectors.toList()));
				} catch (final InvalidRepositoryPathException e) {
					LOG.warn("Error listing paths for {}: {}", currentPath, e.getMessage());
				}

				all.addAll(session.temporaryDirs.stream().filter(path -> path.startsWith(currentPath) && !path.equals(currentPath))
					// we have full paths as string, filter the ones that are direct children
					.map(path -> {
						final Path relativized = root.relativize(Paths.get(path));
						LOG.trace("Rel={} rel[0]={} root.resolve={}", relativized.toString(), relativized.getName(0), root.resolve(relativized.getName(0)).normalize().toString());
						return root.resolve(relativized.getName(0)).normalize().toString();
					})
					// unique
					.distinct().peek(p -> {
						LOG.debug("Temporary folder={}", p);
					})
					// map to RepositoryFtpDirectory
					.map(path -> directory(path, session)).collect(Collectors.toList()));

				// Distinct sorted list of everything
				return Collections.unmodifiableList(all.stream().distinct().sorted((a, b) -> {
					return a.getName().compareTo(b.getName());
				}).peek(path -> {
					// System.err.println("listFiles " + path.getName());
				}).collect(Collectors.toList()));
			}

			@Override
			public String getOwnerName() {
				// TODO Auto-generated method stub
				return "root";
			}

			@Override
			public String getGroupName() {
				// TODO Auto-generated method stub
				return "wheel";
			}

			@Override
			public boolean delete() {
				return FtpRunAs.asFtpUser(session.user, () -> {
					LOG.info("Delete this={}", getAbsolutePath());
					if (session.hasTempDir(getAbsolutePath())) {
						session.removeTempDir(getAbsolutePath());
						return true;
					}
					LOG.warn("Not deleting repository folder={}", getAbsolutePath());
					return false;
				});
			}

			@Override
			public boolean changeWorkingDirectory(final String dir) {
				final String normalized = Paths.get(getAbsolutePath()).resolve(dir).normalize().toAbsolutePath().toString();
				LOG.info("CWD this={} dir={} normalized={}", getAbsolutePath(), dir, normalized);
				this.cwd(normalized);
				// TODO Check if such path exists?
				return true;
			}
		};

		return rfd;
	}
	
	/*
	 * (non-Javadoc)
	 * @see
	 * org.apache.ftpserver.ftplet.FileSystemFactory#createFileSystemView(org.apache
	 * .ftpserver.ftplet.User)
	 */
	@Override
	public FileSystemView createFileSystemView(final User user) throws FtpException {
		LOG.info("Creating new repository view for {}", user.getName());
		
		RepositoryFileSystemView userView = new RepositoryFileSystemView((FtpUser) user) {

			@Override
			public FtpFile getFile(final String file) throws FtpException {
				LOG.debug("getFile file={} for user={}", file, username);
				final Path path = file.startsWith("/") ? Paths.get(file).normalize() : Paths.get(cwd.getAbsolutePath(), file).normalize();
				LOG.trace("Resolved normalized={}", path.toString());
				LOG.trace("Temporary dirs: {}", temporaryDirs);

				if (temporaryDirs.stream().filter(longpath -> longpath.startsWith(path.toString())).findFirst().isPresent()) {
					LOG.trace("dir={} is a temporary session-bound directory", path);
					return directory(path.toString(), this);
				}

				try {
					return isDirectory(path) ?
							// directory
							directory(path.toString(), this) 
							// or file
							: file(FtpRunAs.asFtpUser(user, () -> repositoryService.getFile(
									path.getParent().toString(), 
									path.getFileName().toString())), this);

				} catch (final AuthenticationException e) {
					LOG.warn("Authentication problem {}", e.getMessage(), e);
					throw new AuthenticationFailedException(e.getMessage());
				} catch (final NoSuchRepositoryFileException e) {

					LOG.debug("Making new CanBeAnythingFile path={} name={}", path.getParent().toString(), path.getFileName().toString());
					return new CanBeAnythingFile(path.getParent(), path.getFileName().toString()) {

						@Override
						public boolean mkdir() {
							this.dir = true;
							LOG.info("Mkdir path={}", this.getAbsolutePath());
							temporaryDirs.add(this.getAbsolutePath());
							return true;
						}

						@Override
						public boolean delete() {
							final Set<String> matches = temporaryDirs.stream().filter(longpath -> longpath.startsWith(getAbsolutePath())).collect(Collectors.toSet());
							LOG.debug("Removing session-bound directories {}", matches);
							temporaryDirs.removeAll(matches);
							return true;
						};

						@Override
						public OutputStream createOutputStream(final long offset) throws IOException {
							LOG.info("Creating output stream for new file={} at offset={}", getAbsolutePath(), offset);
							assert (offset == 0l);
							return bytesManager.newFile(user, Paths.get(getAbsolutePath()));
						}
					};
				}
			}

			private boolean isDirectory(final Path path) {
				try {
					return path.toString().equals("/") || repositoryService.hasPath(path) || temporaryDirs.stream().filter(longpath -> longpath.equals(path.toString())).findFirst().isPresent();
				} catch (final InvalidRepositoryPathException e) {
					LOG.debug("Invalid repository path {}: {}", path, e.getMessage());
					return temporaryDirs.stream().filter(longpath -> longpath.equals(path.toString())).findFirst().isPresent();
				}
			}

		};
		
		return userView;
		// AspectJProxyFactory factory = new AspectJProxyFactory(userView);
		// factory.addAspect(new FtpSpringSecurityAspect((FtpUser) user));
		// return factory.getProxy();
	}

	/*
	 * (non-Javadoc)
	 * @see org.springframework.beans.factory.InitializingBean#afterPropertiesSet()
	 */
	@Override
	public void afterPropertiesSet() throws Exception {
		assert (this.repositoryService != null);
		LOG.warn("Initialized RFSF with service={}", this.repositoryService);
	}

	/**
	 * The Class RepositoryFileSystemView.
	 */
	private abstract class RepositoryFileSystemView implements FileSystemView {

		/** The user. */
		protected FtpUser user;

		/** The username. */
		protected String username;

		/** The cwd. */
		protected RepositoryFtpDirectory cwd = directory("/", this);

		/** The home dir. */
		protected RepositoryFtpDirectory homeDir = directory("/", this);

		/** The temporary dirs. */
		protected Set<String> temporaryDirs = new HashSet<>();

		/**
		 * Instantiates a new repository file system view.
		 *
		 * @param user the user
		 */
		public RepositoryFileSystemView(final FtpUser user) {
			this.user = user;
			username = user.getName();
		}

		/**
		 * Removes the temp dir.
		 *
		 * @param absolutePath the absolute path
		 */
		public void removeTempDir(final String absolutePath) {
			final Set<String> matches = temporaryDirs.stream().filter(longpath -> longpath.startsWith(absolutePath)).collect(Collectors.toSet());
			LOG.debug("Removing session-bound directories {}", matches);
			temporaryDirs.removeAll(matches);
		}

		/*
		 * (non-Javadoc)
		 * @see org.apache.ftpserver.ftplet.FileSystemView#isRandomAccessible()
		 */
		@Override
		public boolean isRandomAccessible() throws FtpException {
			// TODO Auto-generated method stub
			return false;
		}

		/*
		 * (non-Javadoc)
		 * @see org.apache.ftpserver.ftplet.FileSystemView#getWorkingDirectory()
		 */
		@Override
		public FtpFile getWorkingDirectory() throws FtpException {
			LOG.debug("getWorkingDirectory for user={}", username);
			return this.cwd;
		}

		/*
		 * (non-Javadoc)
		 * @see org.apache.ftpserver.ftplet.FileSystemView#getHomeDirectory()
		 */
		@Override
		public FtpFile getHomeDirectory() throws FtpException {
			LOG.debug("getHomeDirectory for user={}", username);
			return this.homeDir;
		}

		/*
		 * (non-Javadoc)
		 * @see org.apache.ftpserver.ftplet.FileSystemView#dispose()
		 */
		@Override
		public void dispose() {
			LOG.info("Disposing repository view for user={}", username);
		}

		/*
		 * (non-Javadoc)
		 * @see
		 * org.apache.ftpserver.ftplet.FileSystemView#changeWorkingDirectory(java.lang.
		 * String)
		 */
		@Override
		public boolean changeWorkingDirectory(final String dir) throws FtpException {
			LOG.debug("CWD dir={} for user={}", dir, username);
			return this.cwd.changeWorkingDirectory(dir);
		}

		/**
		 * Checks for temp dir.
		 *
		 * @param absolutePath the absolute path
		 * @return true, if successful
		 */
		public boolean hasTempDir(final String absolutePath) {
			return temporaryDirs.stream().filter(longpath -> longpath.startsWith(absolutePath)).findFirst().isPresent();
		}
	}
}
