/*
 * Copyright 2018 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.filerepository.service.ftp;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Path;
import java.util.List;

import org.apache.ftpserver.ftplet.FtpFile;

/**
 * {@link RepositoryFtpDirectory} wraps repository paths.
 *
 * @author Matija Obreza
 */
public abstract class RepositoryFtpDirectory implements FtpFile {

	/** The path. */
	private Path path;

	/**
	 * Instantiates a new repository ftp directory.
	 *
	 * @param path the path
	 */
	public RepositoryFtpDirectory(final Path path) {
		assert (path != null);
		cwd(path);
	}

	/**
	 * Cwd.
	 *
	 * @param path the path
	 */
	protected void cwd(final Path path) {
		this.path = path.normalize().toAbsolutePath();
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#getAbsolutePath()
	 */
	@Override
	public String getAbsolutePath() {
		return path.normalize().toAbsolutePath().toString();
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#getName()
	 */
	@Override
	public String getName() {
		return path.getFileName().toString();
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#isHidden()
	 */
	@Override
	public boolean isHidden() {
		return false;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#isDirectory()
	 */
	@Override
	public boolean isDirectory() {
		return true;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#isFile()
	 */
	@Override
	public boolean isFile() {
		return false;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#doesExist()
	 */
	@Override
	public boolean doesExist() {
		return true;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#isReadable()
	 */
	@Override
	public boolean isReadable() {
		return true;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#isWritable()
	 */
	@Override
	public boolean isWritable() {
		return true;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#isRemovable()
	 */
	@Override
	public boolean isRemovable() {
		return true;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#getOwnerName()
	 */
	@Override
	public abstract String getOwnerName();

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#getGroupName()
	 */
	@Override
	public abstract String getGroupName();

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#getLinkCount()
	 */
	@Override
	public int getLinkCount() {
		return 0;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#getLastModified()
	 */
	@Override
	public long getLastModified() {
		return 0;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#setLastModified(long)
	 */
	@Override
	public boolean setLastModified(final long time) {
		return false;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#getSize()
	 */
	@Override
	public long getSize() {
		return 0;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#getPhysicalFile()
	 */
	@Override
	public Object getPhysicalFile() {
		// TODO Auto-generated method stub
		return null;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#mkdir()
	 */
	@Override
	public abstract boolean mkdir();

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#delete()
	 */
	@Override
	public abstract boolean delete();

	/*
	 * (non-Javadoc)
	 * @see
	 * org.apache.ftpserver.ftplet.FtpFile#move(org.apache.ftpserver.ftplet.FtpFile)
	 */
	@Override
	public abstract boolean move(FtpFile destination);

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#listFiles()
	 */
	@Override
	public abstract List<? extends FtpFile> listFiles();

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#createOutputStream(long)
	 */
	@Override
	public OutputStream createOutputStream(final long offset) throws IOException {
		throw new IOException("Cannot outputstream the Directory");
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#createInputStream(long)
	 */
	@Override
	public InputStream createInputStream(final long offset) throws IOException {
		throw new IOException("Cannot inputstream the Directory");
	}

	/**
	 * Change working directory.
	 *
	 * @param dir the dir
	 * @return true, if successful
	 */
	public abstract boolean changeWorkingDirectory(String dir);

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = (prime * result) + ((path == null) ? 0 : path.hashCode());
		return result;
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(final Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final RepositoryFtpDirectory other = (RepositoryFtpDirectory) obj;
		if (path == null) {
			if (other.path != null)
				return false;
		} else if (!path.equals(other.path))
			return false;
		return true;
	}
}
