/*
 * Decompiled with CFR 0.152.
 */
package org.genesys.filerepository.service.ftp;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.nio.channels.Channels;
import java.nio.file.Path;
import org.apache.commons.io.IOUtils;
import org.genesys.filerepository.InvalidRepositoryFileDataException;
import org.genesys.filerepository.InvalidRepositoryPathException;
import org.genesys.filerepository.NoSuchRepositoryFileException;
import org.genesys.filerepository.model.RepositoryFile;
import org.genesys.filerepository.service.RepositoryService;
import org.genesys.filerepository.service.ftp.FtpRunAs;
import org.genesys.filerepository.service.ftp.FtpUser;
import org.genesys.filerepository.service.ftp.RepositorySyncOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
public class TemporaryBytesManager {
    private static final Logger LOG = LoggerFactory.getLogger(TemporaryBytesManager.class);
    @Autowired(required=true)
    private RepositoryService repositoryService;

    public OutputStream newFile(final FtpUser user, Path path) throws IOException {
        final Path parent = path.getParent();
        final String filename = path.getFileName().toString();
        LOG.info("Creating new path={} parent={} filename={}", new Object[]{path, parent, filename});
        final File tempFile = File.createTempFile("ftp-", ".data");
        return new RepositorySyncOutputStream(new FileOutputStream(tempFile)){

            @Override
            protected void cleanup() throws IOException {
                LOG.info("Removing temporary file={}", (Object)tempFile.getAbsolutePath());
                tempFile.delete();
            }

            @Override
            protected void synchronizeWithRepository() throws IOException {
                FtpRunAs.asFtpUser(user, () -> {
                    byte[] bytes = TemporaryBytesManager.this.readTempFileToBytes(tempFile);
                    try {
                        LOG.info("Synchronizing file={} with repository path={} originalFilename={}", new Object[]{tempFile.getAbsolutePath(), parent, filename});
                        return TemporaryBytesManager.this.repositoryService.addFile(parent, filename, null, bytes, null);
                    }
                    catch (InvalidRepositoryFileDataException | InvalidRepositoryPathException e) {
                        LOG.warn("Error synchronizing new file parent={} filename={} with repository: {}", new Object[]{parent, filename, e.getMessage()});
                        throw new IOException(e);
                    }
                });
            }
        };
    }

    protected byte[] readTempFileToBytes(File tempFile) throws IOException {
        LOG.trace("Creating memory buffer of size={}", (Object)tempFile.length());
        try (BufferedInputStream tos = new BufferedInputStream(new FileInputStream(tempFile));){
            byte[] byArray;
            try (ByteArrayOutputStream baos = new ByteArrayOutputStream((int)tempFile.length());){
                IOUtils.copy((InputStream)tos, (OutputStream)baos);
                baos.flush();
                byArray = baos.toByteArray();
            }
            return byArray;
        }
    }

    public InputStream createInputStream(RepositoryFile repositoryFile, long offset) throws IOException {
        LOG.info("Preparing input stream for rf={} offset={}", (Object)repositoryFile.getUuid(), (Object)offset);
        final File tempFile = this.shadowRepositoryFile(repositoryFile);
        BufferedInputStream bis = new BufferedInputStream(new FileInputStream(tempFile)){

            @Override
            public void close() throws IOException {
                super.close();
                if (tempFile.exists()) {
                    LOG.debug("Removing temp file={}", (Object)tempFile.getAbsolutePath());
                    tempFile.delete();
                }
            }
        };
        if (offset > 0L) {
            int skipped = 0;
            while ((long)skipped < offset) {
                LOG.trace("Skipping {} of {} bytes to skip in the input stream, skipped {}", new Object[]{offset - (long)skipped, offset, skipped});
                skipped = (int)((long)skipped + bis.skip(offset - (long)skipped));
            }
        }
        return bis;
    }

    private File shadowRepositoryFile(RepositoryFile repositoryFile) throws IOException, FileNotFoundException {
        File tempFile = File.createTempFile("ftp-", ".data");
        byte[] bytes = this.repositoryService.getFileBytes(repositoryFile);
        LOG.debug("Loaded bytes={} from repository file.size={}", (Object)bytes.length, (Object)repositoryFile.getSize());
        try (BufferedOutputStream fos = new BufferedOutputStream(new FileOutputStream(tempFile));){
            IOUtils.write((byte[])bytes, (OutputStream)fos);
            ((OutputStream)fos).flush();
        }
        LOG.debug("Shadow file has length={}", (Object)tempFile.length());
        return tempFile;
    }

    public OutputStream createOutputStream(final FtpUser user, final RepositoryFile repositoryFile, long offset) throws IOException {
        final File tempFile = this.shadowRepositoryFile(repositoryFile);
        LOG.info("Preparing local copy of rf={} offset={} temp={}", new Object[]{repositoryFile.getUuid(), offset, tempFile.getAbsolutePath()});
        final RandomAccessFile raf = new RandomAccessFile(tempFile, "rw");
        if (offset != 0L) {
            LOG.trace("Skipping {} of output stream", (Object)offset);
            raf.seek(offset);
        }
        return new RepositorySyncOutputStream(Channels.newOutputStream(raf.getChannel())){

            @Override
            protected void cleanup() throws IOException {
                LOG.info("Removing temporary file={}", (Object)tempFile.getAbsolutePath());
                IOUtils.closeQuietly((Closeable)raf);
                tempFile.delete();
            }

            @Override
            protected void synchronizeWithRepository() throws IOException {
                FtpRunAs.asFtpUser(user, () -> {
                    byte[] bytes = TemporaryBytesManager.this.readTempFileToBytes(tempFile);
                    LOG.info("Stream to temporary upload buffer is closed, synchronizing bytes={}", (Object)bytes.length);
                    try {
                        TemporaryBytesManager.this.repositoryService.updateBytes(repositoryFile, null, bytes);
                    }
                    catch (NoSuchRepositoryFileException e) {
                        LOG.warn("Error synchronizing new file with repository: {}", (Object)e.getMessage());
                        throw new IOException(e);
                    }
                    LOG.info("Synchronized file={} with repository path={} originalFilename={}", new Object[]{tempFile.getAbsolutePath(), repositoryFile.getFolder().getPath(), repositoryFile.getOriginalFilename()});
                    return null;
                });
            }
        };
    }
}

