/*
 * Copyright 2018 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.filerepository.service.ftp;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;

import org.apache.ftpserver.ftplet.FtpFile;
import org.genesys.filerepository.model.RepositoryFile;

/**
 * {@link RepositoryFtpFile} wraps {@link RepositoryFile}.
 *
 * @author Matija Obreza
 */
public abstract class RepositoryFtpFile implements FtpFile {

	/** The repository file. */
	private final RepositoryFile repositoryFile;

	/**
	 * Instantiates a new repository ftp file.
	 *
	 * @param repositoryFile the repository file
	 */
	public RepositoryFtpFile(final RepositoryFile repositoryFile) {
		assert (repositoryFile != null);
		this.repositoryFile = repositoryFile;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#getAbsolutePath()
	 */
	@Override
	public String getAbsolutePath() {
		return this.repositoryFile.getFolder().getPath() + this.repositoryFile.getOriginalFilename();
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#getName()
	 */
	@Override
	public String getName() {
		return this.repositoryFile.getOriginalFilename();
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#isHidden()
	 */
	@Override
	public boolean isHidden() {
		return false;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#isDirectory()
	 */
	@Override
	public boolean isDirectory() {
		return false;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#isFile()
	 */
	@Override
	public boolean isFile() {
		return true;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#doesExist()
	 */
	@Override
	public boolean doesExist() {
		return true;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#isReadable()
	 */
	@Override
	public boolean isReadable() {
		return true;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#isWritable()
	 */
	@Override
	public boolean isWritable() {
		return true;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#isRemovable()
	 */
	@Override
	public boolean isRemovable() {
		return true;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#getOwnerName()
	 */
	@Override
	public abstract String getOwnerName();

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#getGroupName()
	 */
	@Override
	public abstract String getGroupName();

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#getLinkCount()
	 */
	@Override
	public int getLinkCount() {
		return 0;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#getLastModified()
	 */
	@Override
	public long getLastModified() {
		return repositoryFile.getLastModifiedDate() != null ? repositoryFile.getLastModifiedDate().getEpochSecond() * 1000 : 0;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#setLastModified(long)
	 */
	@Override
	public boolean setLastModified(final long time) {
		return false;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#getSize()
	 */
	@Override
	public long getSize() {
		return repositoryFile.getSize();
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#getPhysicalFile()
	 */
	@Override
	public Object getPhysicalFile() {
		System.err.println("getPhysicalFile??");
		return null;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#mkdir()
	 */
	@Override
	public abstract boolean mkdir();

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#delete()
	 */
	@Override
	public abstract boolean delete();

	/*
	 * (non-Javadoc)
	 * @see
	 * org.apache.ftpserver.ftplet.FtpFile#move(org.apache.ftpserver.ftplet.FtpFile)
	 */
	@Override
	public abstract boolean move(FtpFile destination);

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#listFiles()
	 */
	@Override
	public List<? extends FtpFile> listFiles() {
		return null;
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#createOutputStream(long)
	 */
	@Override
	public abstract OutputStream createOutputStream(long offset) throws IOException;

	/*
	 * (non-Javadoc)
	 * @see org.apache.ftpserver.ftplet.FtpFile#createInputStream(long)
	 */
	@Override
	public abstract InputStream createInputStream(long offset) throws IOException;

}
