/*
 * Copyright 2016 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.geotools.service;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.ArrayUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Helper for CSV header management.
 */
public abstract class HeaderUtils {

	/** The Constant LOG. */
	private final static Logger LOG = LoggerFactory.getLogger(HeaderUtils.class);

	/**
	 * Check for invalid headers and throw IOException if one is found.
	 *
	 * @param headers input headers
	 * @param notAllowedHeaders array of unacceptable headers
	 * @throws IOException when one unacceptable header is found
	 */
	public static void throwIfHeaderFound(final String[] headers, final String[] notAllowedHeaders) throws IOException {
		for (final String header : notAllowedHeaders) {
			final int pos = ArrayUtils.indexOf(headers, header);
			if (pos >= 0) {
				throw new IOException("Header " + header + " found in input file. Refusing to run.");
			}
		}
	}

	/**
	 * Check for required headers and throw IOException if one is missing.
	 *
	 * @param headers input headers
	 * @param requiredHeaders the array of required headers
	 * @throws IOException when one unacceptable header is found
	 */
	public static void throwIfHeaderNotFound(final String[] headers, final String[] requiredHeaders) throws IOException {
		for (final String header : requiredHeaders) {
			final int pos = ArrayUtils.indexOf(headers, header);
			if (pos < 0) {
				throw new IOException("Header " + header + " not found in input file. Refusing to run.");
			}
		}
	}

	/**
	 * Map CSV headers to their positions.
	 *
	 * @param headers CSV header list
	 * @return a new List with header indexes
	 */
	public static List<Integer> mapHeaderPositions(final String[] headers) {
		final List<Integer> outputMapping = new ArrayList<>();
		int col = 0;
		for (final String header : headers) {
			LOG.debug("Mapping input column={} to index={}", header, col);
			outputMapping.add(col++);
		}
		return outputMapping;
	}

	/**
	 * We are mostly interested in only a few supported headers. Get their indexes.
	 *
	 * @param headers the headers
	 * @param supportedHeaders Array of supported headers
	 * @return Map of positions of supported headers
	 */
	public static Map<String, Integer> makeSourceMapping(final String[] headers, final String[] supportedHeaders) {
		final Map<String, Integer> sourceMapping = new HashMap<>();

		Arrays.stream(supportedHeaders).forEach(header -> {
			LOG.debug("Looking for header {} in {}", header, Arrays.asList(headers));
			final int pos = ArrayUtils.indexOf(headers, header);
			if (pos >= 0) {
				LOG.debug("Header {} is in position {}", header, pos);
				sourceMapping.put(header, pos);
			} else {
				LOG.info("No header {}", header);
			}
		});

		return sourceMapping;
	}

	/**
	 * Copy data from input CSV line to output CSV line with appropriate destination mapping.
	 *
	 * @param nextLine source line
	 * @param outputHeaders output headers
	 * @param outputMapping mapping of output columns
	 * @return the output CSV line filled with data from input CSV line
	 */
	public static String[] toOutputLine(final String[] nextLine, final List<String> outputHeaders, final List<Integer> outputMapping) {
		final String[] outputLine = new String[outputHeaders.size()];
		for (int outputCol = 0; outputCol < outputMapping.size(); outputCol++) {
			final Integer mapped = outputMapping.get(outputCol);
			if (mapped != null)
				outputLine[outputCol] = nextLine[mapped];
		}
		return outputLine;
	}

}
