/*
 * Copyright 2016 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.genesys.geotools.service;

import java.io.Serializable;

/**
 * LonLatCacheKey is used as caching key.
 */
public class LonLatCacheKey implements Serializable {

	/** The Constant serialVersionUID. */
	private static final long serialVersionUID = -3626533849742141104L;

	/** The longitude. */
	private final float longitude;

	/** The latitude. */
	private final float latitude;

	/** The orig cty. */
	private String origCty;

	/** The allowed distance margin. */
	private final int allowedDistanceMargin;

	/**
	 * Instantiates a new lon lat cache key.
	 *
	 * @param longitude the longitude
	 * @param latitude the latitude
	 * @param allowedDistanceMargin the allowed distance margin
	 */
	public LonLatCacheKey(final float longitude, final float latitude, final int allowedDistanceMargin) {
		this.longitude = longitude;
		this.latitude = latitude;
		this.allowedDistanceMargin = allowedDistanceMargin;
	}

	/**
	 * Instantiates a new lon lat cache key.
	 *
	 * @param longitude the longitude
	 * @param latitude the latitude
	 * @param origCty the orig cty
	 * @param allowedDistanceMargin the allowed distance margin
	 */
	public LonLatCacheKey(final float longitude, final float latitude, final String origCty, final int allowedDistanceMargin) {
		this.longitude = longitude;
		this.latitude = latitude;
		this.origCty = origCty;
		this.allowedDistanceMargin = allowedDistanceMargin;
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = (prime * result) + allowedDistanceMargin;
		result = (prime * result) + Float.floatToIntBits(latitude);
		result = (prime * result) + Float.floatToIntBits(longitude);
		result = (prime * result) + ((origCty == null) ? 0 : origCty.hashCode());
		return result;
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(final Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final LonLatCacheKey other = (LonLatCacheKey) obj;
		if (allowedDistanceMargin != other.allowedDistanceMargin)
			return false;
		if (Float.floatToIntBits(latitude) != Float.floatToIntBits(other.latitude))
			return false;
		if (Float.floatToIntBits(longitude) != Float.floatToIntBits(other.longitude))
			return false;
		if (origCty == null) {
			if (other.origCty != null)
				return false;
		} else if (!origCty.equals(other.origCty))
			return false;
		return true;
	}

	/**
	 * Gets the longitude.
	 *
	 * @return the longitude
	 */
	public float getLongitude() {
		return longitude;
	}

	/**
	 * Gets the latitude.
	 *
	 * @return the latitude
	 */
	public float getLatitude() {
		return latitude;
	}

	/**
	 * Gets the allowed distance margin.
	 *
	 * @return the allowed distance margin
	 */
	public int getAllowedDistanceMargin() {
		return allowedDistanceMargin;
	}

	/**
	 * Gets the orig cty.
	 *
	 * @return the orig cty
	 */
	public String getOrigCty() {
		return origCty;
	}
}
