/*
 * Copyright 2016 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.genesys.geotools.service;

import java.awt.geom.Point2D;
import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.util.HashMap;
import java.util.Map;

import org.geotools.data.DataStore;
import org.geotools.data.DataStoreFinder;
import org.geotools.geometry.jts.JTSFactoryFinder;
import org.geotools.referencing.GeodeticCalculator;
import org.geotools.referencing.crs.DefaultGeographicCRS;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.Polygon;
import org.opengis.geometry.MismatchedDimensionException;
import org.opengis.referencing.FactoryException;
import org.opengis.referencing.operation.TransformException;


/**
 * The Class ShapefileUtils.
 */
public class ShapefileUtils {

	/** The Constant RING_POINTS. */
	private static final int RING_POINTS = 12;

	/** The Constant geometryFactory. */
	private static final GeometryFactory geometryFactory = JTSFactoryFinder.getGeometryFactory(null);

	/**
	 * Open shape file.
	 *
	 * @param shapeFile the shape file
	 * @return the data store
	 * @throws MalformedURLException the malformed url exception
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public static DataStore openShapeFile(final File shapeFile) throws MalformedURLException, IOException {

		final Map<String, Object> map = new HashMap<>();
		map.put("url", shapeFile.toURI().toURL());
		System.err.println(shapeFile.toURI().toURL());

		final DataStore dataStore = DataStoreFinder.getDataStore(map);
		return dataStore;
	}

	/**
	 * Uses ideas from https://geoaware.wordpress.com/2013/10/16/geodetic-buffers-with-geotools/
	 *
	 * @param longitude the longitude
	 * @param latitude the latitude
	 * @param distanceMeters the distance meters
	 * @return the point buffer
	 * @throws MismatchedDimensionException the mismatched dimension exception
	 * @throws TransformException the transform exception
	 * @throws FactoryException the factory exception
	 */
	public static Geometry getPointBuffer(final double longitude, final double latitude, final double distanceMeters)
			throws MismatchedDimensionException, TransformException, FactoryException {

		final Coordinate[] coords1 = new Coordinate[RING_POINTS + 1];

		final GeodeticCalculator gc = new GeodeticCalculator(DefaultGeographicCRS.WGS84);
		gc.setStartingGeographicPoint(longitude, latitude);

		for (int i = 0; i < RING_POINTS; i++) {
			gc.setDirection((i * 360.0) / RING_POINTS, distanceMeters);
			final Point2D destGeoPoint = gc.getDestinationGeographicPoint();
			coords1[i] = new Coordinate(destGeoPoint.getX(), destGeoPoint.getY(), 0);
			// System.err.println("a=" + (i * 360.0 / RING_POINTS) + " p=" + p +
			// " d=" + gc.getOrthodromicDistance());
		}
		// Points of LinearRing form a closed linestring
		coords1[RING_POINTS] = coords1[0];

		final Polygon x = geometryFactory.createPolygon(geometryFactory.createLinearRing(coords1));
		// System.err.println(x);
		return x;
	}

}
