/*
 * Copyright 2016 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.genesys.geotools.service.impl;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.StopWatch;
import org.genesys.geotools.service.CountryOfOriginService;
import org.genesys.geotools.service.LonLatCacheKey;
import org.genesys.geotools.service.ShapefileUtils;
import org.geotools.data.DataStore;
import org.geotools.data.FeatureSource;
import org.geotools.factory.CommonFactoryFinder;
import org.geotools.feature.FeatureCollection;
import org.geotools.feature.FeatureIterator;
import org.geotools.geometry.jts.JTS;
import org.geotools.geometry.jts.JTSFactoryFinder;
import org.geotools.geometry.jts.ReferencedEnvelope;
import org.geotools.referencing.GeodeticCalculator;
import org.geotools.referencing.crs.DefaultGeographicCRS;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.MultiPolygon;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.operation.distance.DistanceOp;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;
import org.opengis.filter.Filter;
import org.opengis.filter.FilterFactory2;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.CacheStats;
import com.google.common.cache.LoadingCache;

/**
 * Polygons used here are derived from OSM data, © OpenStreetMap contributors.
 */
public class CountryOfOriginServiceImpl implements CountryOfOriginService {

	/** LOG */
	private final static Logger LOG = LoggerFactory.getLogger(CountryOfOriginServiceImpl.class);

	/** The Constant ff. */
	private static final FilterFactory2 ff = CommonFactoryFinder.getFilterFactory2();

	/** The Constant geometryFactory. */
	private static final GeometryFactory geometryFactory = JTSFactoryFinder.getGeometryFactory();

	/** The source admin0. */
	private FeatureSource<SimpleFeatureType, SimpleFeature> sourceAdmin0;

	/** The source admin0 x. */
	private FeatureSource<SimpleFeatureType, SimpleFeature> sourceAdmin0X;

	/** The country cache. */
	private LoadingCache<LonLatCacheKey, String> countryCache;

	/** The country cache. */
	private LoadingCache<LonLatCacheKey, Double> countryDistanceCache;

	/** The debug. */
	private final boolean debug = false;

	private String dataFolderPath = "data";

	public void setDataFolderPath(String dataFolderPath) {
		this.dataFolderPath = dataFolderPath;
	}

	static {
		try {
			// Initialize stuff

		} catch (final Throwable e) {
			// big problem
			throw new RuntimeException(e);
		}
	}

	/**
	 * After properties set.
	 *
	 * @throws MalformedURLException the malformed url exception
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public void afterPropertiesSet() throws MalformedURLException, IOException {
		// http://www.gadm.org/version2
		// Use "six dissolved layers"
		final DataStore dataStoreAdm0 = ShapefileUtils.openShapeFile(new File(dataFolderPath, "TM_WORLD_BORDERS-0.3.shp"));
		final DataStore dataStoreAdm0X = ShapefileUtils.openShapeFile(new File(dataFolderPath, "gadm28_adm1.shp"));

		sourceAdmin0 = dataStoreAdm0.getFeatureSource(dataStoreAdm0.getTypeNames()[0]);
		sourceAdmin0X = dataStoreAdm0X.getFeatureSource(dataStoreAdm0X.getTypeNames()[0]);

		countryCache = CacheBuilder.newBuilder().maximumSize(5000).recordStats().expireAfterWrite(20, TimeUnit.SECONDS).build(new CacheLoader<LonLatCacheKey, String>() {
			@Override
			public String load(final LonLatCacheKey key) throws Exception {
				// LOG.debug("Loading");
				return _getCountry(key.getLongitude(), key.getLatitude(), key.getOrigCty(), key.getAllowedDistanceMargin());
			}
		});

		countryDistanceCache = CacheBuilder.newBuilder().maximumSize(5000).recordStats().expireAfterWrite(20, TimeUnit.SECONDS).build(new CacheLoader<LonLatCacheKey, Double>() {
			@Override
			public Double load(final LonLatCacheKey key) throws Exception {
				// LOG.debug("Loading");
				return _distanceToBorder(key.getLongitude(), key.getLatitude(), key.getOrigCty());
			}
		});
	}

	/*
	 * (non-Javadoc)
	 * @see org.genesys.geotools.LandOrSeaService#isOnLand(float, float, int)
	 */
	@Override
	public String getCountries(final float longitude, final float latitude, final String origCty, final int allowedDistanceMargin) throws Exception {

		// 1 geographical mile is 1855.3248 metres for WGS84
		// 1855.3248m * 60 = 111319.488m
		// 111319.488m for 1 degree, can trim by 1000 for 100m precision at
		// equator

		// longitude = (long) (longitude * 1000) / 1000f;
		// latitude = (long) (latitude * 1000) / 1000f;

		try {
			return countryCache.get(new LonLatCacheKey(longitude, latitude, origCty, allowedDistanceMargin));
		} catch (final ExecutionException e) {
			throw new Exception(e.getCause());
		}
	}

	/**
	 * _get country.
	 *
	 * @param longitude the longitude
	 * @param latitude the latitude
	 * @param origCtyISO the orig cty iso
	 * @param allowedDistanceMargin the allowed distance margin
	 * @return the string
	 * @throws Exception the exception
	 */
	private String _getCountry(final float longitude, final float latitude, final String origCtyISO, final int allowedDistanceMargin) throws Exception {

		// LOG.debug(longitude + ", " + latitude + " " + origCtyISO);
		final StopWatch stopWatch = new StopWatch();

		final Point point = geometryFactory.createPoint(new Coordinate(longitude, latitude));
		final String geometryPropertyName = sourceAdmin0.getSchema().getGeometryDescriptor().getLocalName();
		// CoordinateReferenceSystem targetCRS =
		// sourceAdmin0.getSchema().getGeometryDescriptor()
		// .getCoordinateReferenceSystem();

		final ReferencedEnvelope bbox = new ReferencedEnvelope(longitude - 30, longitude + 30, latitude - 30, latitude + 30, DefaultGeographicCRS.WGS84);

		final Filter filterExact =
				// ff.and(
				// ff.equal(ff.property("ISO3"), ff.literal(origCtyISO), false),
				ff.and(ff.bbox(ff.property(geometryPropertyName), bbox), ff.contains(ff.property(geometryPropertyName), ff.literal(point)));

		final StringBuffer sb = new StringBuffer();

		stopWatch.start();
		// LOG.debug(1);
		final FeatureCollection<SimpleFeatureType, SimpleFeature> matchingFeatures = sourceAdmin0.getFeatures(filterExact);
		try (FeatureIterator<SimpleFeature> features = matchingFeatures.features()) {
			if (features.hasNext()) {
				// LOG.debug(3);
				final SimpleFeature feature = features.next();
				// System.out.print(feature.getID());
				// System.out.print(": ");
				// System.out.println(feature.getDefaultGeometryProperty().getValue());
				if (sb.length() > 0)
					sb.append(", ");
				sb.append(feature.getAttribute("ISO3"));
			}
		}

		stopWatch.split();
		long processingTime = stopWatch.getSplitTime();
		if (debug) {
			LOG.debug("Processing time split: " + processingTime);
		}

		if ((sb.length() == 0) || StringUtils.trimToNull(origCtyISO) == null || !sb.toString().contains(origCtyISO)) {
			// if (sb.length() > 0) sb.append(", ");
			// sb.append("???");
			// Filter filterBuffered =
			// ff.intersects(ff.property(geometryPropertyName),
			// ff.literal(ShapefileUtils.getPointBuffer(point,
			// allowedDistanceMargin)));
			// LOG.debug(4);
			final FeatureCollection<SimpleFeatureType, SimpleFeature> matchingFeatures2 = sourceAdmin0X.getFeatures(filterExact);
			// LOG.debug(4.1);

			try (FeatureIterator<SimpleFeature> features = matchingFeatures2.features()) {
				if (features.hasNext()) {
					// LOG.debug(7);
					final SimpleFeature feature = features.next();
					// System.out.print(feature.getID());
					// System.out.print(": ");
					//
					// System.out.println(feature.getDefaultGeometryProperty().getValue());
					final Object isoAttr = feature.getAttribute("ISO");

					final String countryIsoCode = isoAttr == null ? "" : isoAttr.toString();
					if (!sb.toString().contains(countryIsoCode)) {
						if (sb.length() > 0)
							sb.append(", ");
						sb.append(countryIsoCode);
					}
				}
			}
		}
		stopWatch.stop();
		processingTime = stopWatch.getTime();
		if (LOG.isInfoEnabled() || (processingTime > 3000)) {
			LOG.info("Total processing time: {} for\t{}\t\t", processingTime, origCtyISO, longitude, latitude);
		}

		// LOG.debug(sb);
		return sb.toString();
	}

	@Override
	public double distanceToBorder(float longitude, float latitude, String origCty) throws Exception {
		try {
			return countryDistanceCache.get(new LonLatCacheKey(longitude, latitude, origCty, 0));
		} catch (final ExecutionException e) {
			throw new Exception(e.getCause());
		}
	}

	private double _distanceToBorder(float longitude, float latitude, String origCty) throws Exception {
		Coordinate coordinate = new Coordinate(longitude, latitude);
		final Point point = geometryFactory.createPoint(coordinate);

		final Filter filterExact = ff.equal(ff.property("ISO3"), ff.literal(origCty), false);
		LOG.trace("Distance of {} to {}", coordinate, origCty);

		try {
			final FeatureCollection<SimpleFeatureType, SimpleFeature> matchingFeatures = sourceAdmin0.getFeatures(filterExact);
			if (matchingFeatures.size() > 0) {
				if (LOG.isTraceEnabled()) {
					try (FeatureIterator<SimpleFeature> features = matchingFeatures.features()) {
						while (features.hasNext()) {
							final SimpleFeature feature = features.next();
							LOG.trace("{}: {} Attrs={}", feature.getID(), feature.getDefaultGeometryProperty().getValue(), feature.getAttributes());
						}
					}
				}
				try (FeatureIterator<SimpleFeature> features = matchingFeatures.features()) {
					if (features.hasNext()) {
						final SimpleFeature feature = features.next();
						MultiPolygon geometry = (MultiPolygon) feature.getDefaultGeometry();
						GeodeticCalculator gc = new GeodeticCalculator(DefaultGeographicCRS.WGS84);
						gc.setStartingPosition(JTS.toDirectPosition(DistanceOp.nearestPoints(geometry, point)[0], DefaultGeographicCRS.WGS84));
						gc.setDestinationPosition(JTS.toDirectPosition(coordinate, DefaultGeographicCRS.WGS84));

						return gc.getOrthodromicDistance();
					} else {
						return -1.0d;
					}
				}

			} else {
				LOG.debug("No geometry with ISO={}", origCty);
				return -1.0d;
			}
		} catch (IOException e) {
			throw e;
		}
	}

	/**
	 * Prints the cache.
	 */
	public void printCache() {
		final CacheStats stats = countryCache.stats();
		if (LOG.isInfoEnabled()) {
			LOG.info("Hit count=" + stats.hitCount() + " rate=" + stats.hitRate() + "  Miss count=" + stats.missCount() + " rate=" + stats.missRate());
		}
	}
}
