/*
 * Copyright 2016 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.genesys.geotools.service.impl;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

import org.genesys.geotools.service.LandOrSeaService;
import org.genesys.geotools.service.LonLatCacheKey;
import org.genesys.geotools.service.ShapefileUtils;
import org.geotools.data.DataStore;
import org.geotools.data.FeatureSource;
import org.geotools.factory.CommonFactoryFinder;
import org.geotools.feature.FeatureCollection;
import org.geotools.feature.FeatureIterator;
import org.geotools.geometry.jts.JTSFactoryFinder;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.Point;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;
import org.opengis.filter.Filter;
import org.opengis.filter.FilterFactory2;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;

/**
 * Polygons used here are derived from OSM data, © OpenStreetMap contributors.
 */
public class LandOrSeaServiceImpl implements LandOrSeaService {

	/** LOG */
	private final static Logger LOG = LoggerFactory.getLogger(LandOrSeaServiceImpl.class);

	/** The Constant ff. */
	private static final FilterFactory2 ff = CommonFactoryFinder.getFilterFactory2();

	/** The Constant geometryFactory. */
	private static final GeometryFactory geometryFactory = JTSFactoryFinder.getGeometryFactory();

	/** The source land. */
	private FeatureSource<SimpleFeatureType, SimpleFeature> sourceLand;

	/** The source water. */
	private FeatureSource<SimpleFeatureType, SimpleFeature> sourceWater;

	/** The water cache. */
	private LoadingCache<LonLatCacheKey, Boolean> waterCache;

	/** The classifier cache. */
	private LoadingCache<LonLatCacheKey, String> classifierCache;

	private String dataFolderPath = "data";
	
	public void setDataFolderPath(String dataFolderPath) {
		this.dataFolderPath = dataFolderPath;
	}

	/**
	 * After properties set.
	 *
	 * @throws MalformedURLException the malformed url exception
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public void afterPropertiesSet() throws MalformedURLException, IOException {
		// http://openstreetmapdata.com/data/land-polygons
		// Use "split large polygons"
		final DataStore dataStoreLand = ShapefileUtils.openShapeFile(new File(dataFolderPath, "land_polygons.shp"));

		// http://openstreetmapdata.com/data/water-polygons
		// Use "split large polygons"
		final DataStore dataStoreWater = ShapefileUtils.openShapeFile(new File(dataFolderPath, "water_polygons.shp"));

		sourceLand = dataStoreLand.getFeatureSource(dataStoreLand.getTypeNames()[0]);
		sourceWater = dataStoreWater.getFeatureSource(dataStoreWater.getTypeNames()[0]);

		waterCache = CacheBuilder.newBuilder().maximumSize(1000).expireAfterWrite(5, TimeUnit.SECONDS).build(
			new CacheLoader<LonLatCacheKey, Boolean>() {
				@Override
				public Boolean load(final LonLatCacheKey key) throws Exception {
					// LOG.debug("Loading");
					return _isOnLand(key.getLongitude(), key.getLatitude(), key.getAllowedDistanceMargin());
				}
			});

		classifierCache = CacheBuilder.newBuilder().maximumSize(1000).expireAfterWrite(5, TimeUnit.SECONDS).build(
			new CacheLoader<LonLatCacheKey, String>() {
				@Override
				public String load(final LonLatCacheKey key) throws Exception {
					// LOG.debug("Loading");
					return _classifyLocation(key.getLongitude(), key.getLatitude(), key.getAllowedDistanceMargin());
				}
			});
	}

	/**
	 * Classify location.
	 *
	 * @param longitude the longitude
	 * @param latitude the latitude
	 * @param allowedDistanceMargin the allowed distance margin
	 * @return the string
	 * @throws Exception the exception
	 */
	@Override
	public String classifyLocation(final float longitude, final float latitude, final int allowedDistanceMargin) throws Exception {
		try {
			return classifierCache.get(new LonLatCacheKey(longitude, latitude, allowedDistanceMargin));
		} catch (final ExecutionException e) {
			throw new Exception(e.getCause());
		}
	}

	/**
	 * _classify location.
	 *
	 * @param longitude the longitude
	 * @param latitude the latitude
	 * @param allowedDistanceMargin the allowed distance margin
	 * @return the string
	 * @throws Exception the exception
	 */
	private String _classifyLocation(final float longitude, final float latitude, final int allowedDistanceMargin) throws Exception {

		final Point point = geometryFactory.createPoint(new Coordinate(longitude, latitude));

		final Filter filterExact = ff.contains(ff.property("the_geom"), ff.literal(point));

		try {
			if (sourceWater.getFeatures(filterExact).size() > 0) {

				// Create a buffer and check for Land
				final Filter filterBuffered = ff.intersects(ff.property("the_geom"), ff.literal(ShapefileUtils.getPointBuffer(longitude, latitude,
					allowedDistanceMargin)));

				final FeatureCollection<SimpleFeatureType, SimpleFeature> matchingFeatures = sourceLand.getFeatures(filterBuffered);
				if (matchingFeatures.size() > 0) {
					if (LOG.isTraceEnabled()) {
						try (FeatureIterator<SimpleFeature> features = matchingFeatures.features()) {
							while (features.hasNext()) {
								final SimpleFeature feature = features.next();
								LOG.trace("{}: {} Attrs={}", feature.getID(), feature.getDefaultGeometryProperty().getValue(), feature.getAttributes());
							}
						}
					}

					return "Coastal";
				} else {
					return "Water";
				}
			} else if (sourceLand.getFeatures(filterExact).size() > 0) {
				// // Create a buffer and check for water
				// Filter filterBuffered =
				// ff.intersects(ff.property("the_geom"),
				// ff.literal(getPointBuffer(point, allowedDistanceFromLand)));
				//
				// if (sourceWater.getFeatures(filterBuffered).size() > 0) {
				// return "Coastal";
				// } else {
				return "Land";
				// }
			} else {
				throw new Exception("Can't figure out the result for (" + longitude + ", " + latitude + ") with distance=" + allowedDistanceMargin);
			}

		} catch (final Exception e) {
			throw e;
		}
	}

	/*
	 * (non-Javadoc)
	 * @see org.genesys.geotools.LandOrSeaService#isOnLand(float, float, int)
	 */
	@Override
	public Boolean isOnLand(final float longitude, final float latitude, final int allowedDistanceFromLand) throws Exception {
		try {
			return waterCache.get(new LonLatCacheKey(longitude, latitude, allowedDistanceFromLand));
		} catch (final ExecutionException e) {
			throw new Exception(e.getCause());
		}
	}

	/**
	 * _is on land.
	 *
	 * @param longitude the longitude
	 * @param latitude the latitude
	 * @param allowedDistanceFromLand the allowed distance from land
	 * @return the boolean
	 * @throws Exception the exception
	 */
	private Boolean _isOnLand(final float longitude, final float latitude, final int allowedDistanceFromLand) throws Exception {

		final Point point = geometryFactory.createPoint(new Coordinate(longitude, latitude));

		final Filter filterExact = ff.contains(ff.property("the_geom"), ff.literal(point));

		try {
			if (sourceWater.getFeatures(filterExact).size() > 0) {

				// Create a buffer and check for Land
				final Filter filterBuffered = ff.intersects(ff.property("the_geom"), ff.literal(ShapefileUtils.getPointBuffer(longitude, latitude,
					allowedDistanceFromLand)));

				if (sourceLand.getFeatures(filterBuffered).size() > 0) {
					return true;
				} else {
					return false;
				}
			} else if (sourceLand.getFeatures(filterExact).size() > 0) {
				return true;
			} else {
				throw new Exception("Can't figure out the result for " + longitude + "," + latitude + " with distance=" + allowedDistanceFromLand);
			}

		} catch (final Exception e) {
			throw e;
		}
	}

}
