/*
 * Copyright 2015 Global Crop Diversity Trust
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.genesys2.gringlobal.taxonomy.component;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Suggest fixes to scientific names based on an in-memory list of valid taxonomies.
 */
public class TaxonomyChecker {

	/** The Constant LOG. */
	private final static Logger LOG = LoggerFactory.getLogger(TaxonomyChecker.class);

	/** The database. */
	private TaxonomyDatabase database;

	/**
	 * Sets the taxonomy database.
	 *
	 * @param database the new taxonomy database
	 */
	public void setTaxonomyDatabase(TaxonomyDatabase database) {
		this.database = database;
	}

	/**
	 * Find suggestions for GENUS.
	 *
	 * @param genus the genus
	 * @param maxSize the max size
	 * @return suggested fixes for genus or empty list when genus is fine or when there are no suggestions.
	 */
	public List<String> suggestGenus(String genus, int maxSize) {
		if (StringUtils.isBlank(genus) || StringUtils.equals("Unknown", genus)) {
			return Arrays.asList("Unknown");
		} else if (database.containsGenus(genus)) {
			LOG.debug("Database contains genus={}", genus);
			return Collections.emptyList();
		} else {
			LOG.debug("Database does not contain genus={}", genus);
		}
		return database.findSimilarGenus(genus, maxSize);
	}

	/**
	 * Find suggestions for SPECIES.
	 *
	 * @param genus the genus
	 * @param species the species
	 * @param maxSize TODO
	 * @return suggested fixes for genus or empty list if species is fine or when there are no suggestions.
	 */
	public List<String> suggestSpecies(String genus, String species, int maxSize) {
		LOG.debug("Suggesting species for genus={} species={}", genus, species);
		if (database.containsSpecies(genus, species)) {
			return Collections.emptyList();
		}

		if (database.containsGenus(genus)) {
			// Genus exists
			LOG.debug("Database contains genus={}", genus);
			return database.findSimilarSpecies(genus, species, maxSize);
		} else {
			LOG.debug("Database does not contain genus={}", genus);
		}

		// Genus not listed in the database, go through suggestions
		List<String> suggestions = new ArrayList<>(0);
		for (String suggestedGenus : suggestGenus(genus, 2)) {
			LOG.debug("Suggesting with suggested genus={}", suggestedGenus);
			suggestions.addAll(database.findSimilarSpecies(suggestedGenus, species, 2).stream()
					.map(suggestion -> suggestedGenus.equalsIgnoreCase(genus) ? suggestion : "(" + suggestedGenus + ") " + suggestion).collect(Collectors.toList()));
		}
		return suggestions;
	}

	/**
	 * Return species authority for the genus + species.
	 *
	 * @param genus the genus
	 * @param species the species
	 * @return species authority if there's one matching record in the database
	 */
	public String getSpeciesAuthority(String genus, String species) {
		if (!database.containsSpecies(genus, species)) {
			return null;
		}

		return database.getSpeciesAuthority(genus, species);
	}
}
