/*
 * Copyright 2015 Global Crop Diversity Trust
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.genesys2.gringlobal.taxonomy.component;

import java.util.List;

/**
 * Defines searches possible against a database of taxonomic data.
 */
public interface TaxonomyDatabase {

	/**
	 * Does the database contain at least one record that exactly matches specified genus?.
	 *
	 * @param genus the genus
	 * @return <code>true</code> if there is at least one record
	 */
	boolean containsGenus(String genus);

	/**
	 * Return a list of genera close to the specified genus.
	 * 
	 * @param genus genus
	 * @param maxSize TODO
	 * @return list of suggestions, ordered by preference (best first), never null.
	 */
	List<String> findSimilarGenus(String genus, int maxSize);

	/**
	 * Does the database contain at least one record that exactly matches specified genus and species?.
	 *
	 * @param genus the genus
	 * @param species the species
	 * @return <code>true</code> if there is at least one record
	 */
	boolean containsSpecies(String genus, String species);

	/**
	 * Return a list of species close to the specified genus and species.
	 *
	 * @param genus valid genus
	 * @param species species
	 * @param maxSize TODO
	 * @return list of suggestions, ordered by preference (best first), never null.
	 */
	List<String> findSimilarSpecies(String genus, String species, int maxSize);

	/**
	 * Return the species authority for the specified species within genus.
	 *
	 * @param genus the genus
	 * @param species the species
	 * @return the authority or <code>null</code>
	 */
	String getSpeciesAuthority(String genus, String species);

}
