/*
 * Copyright 2015 Global Crop Diversity Trust
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys2.gringlobal.taxonomy.component;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.Reader;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

import com.opencsv.CSVParser;
import com.opencsv.CSVReader;

import org.genesys2.gringlobal.taxonomy.model.GenusRow;
import org.genesys2.gringlobal.taxonomy.model.SpeciesRow;

/**
 * The CSV file reader for GRIN-Global taxonomy.
 */
public class TaxonomyReader {

	/** The Constant sdf. */
	private static final SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

	/**
	 * Returns a {@link CSVReader} that properly handles GRIN-Global taxonomy files in CSV format. It converts \N to <code>null</code> in each row.
	 *
	 * @param reader the reader
	 * @param startAt the start at
	 * @return the CSV reader
	 */
	public static CSVReader openCsvReader(Reader reader, int startAt) {
		CSVReader csvReader = new CSVReader(new BufferedReader(reader), startAt, new CSVParser('\t', '"', (char) 0, false, false, true) {
			@Override
			protected String[] parseLine(String nextLine, boolean multi) throws IOException {
				String[] res = super.parseLine(nextLine, multi);
				if (res != null)
					for (int i = 0; i < res.length; i++) {
						if ("\\N".equals(res[i])) {
							res[i] = null;
						}
					}
				return res;
			}
		});

		return csvReader;
	}

	/**
	 * Convert row to GenusRow.
	 *
	 * @param row in the right CSV order
	 * @return the genus row
	 * @throws ParseException if date cannot be parsed
	 */
	public static GenusRow toGenus(String[] row) throws ParseException {
		if (row == null || row.length == 0) {
			return null;
		}

		GenusRow genusRow = new GenusRow();

		genusRow.setGenusId(toLong(row[0]));
		genusRow.setCurrentGenusId(toLong(row[1]));
		genusRow.setFamilyId(toLong(row[2]));
		genusRow.setQualifyingCode(row[3]);
		genusRow.setHybrid(toBoolean(row[4]));
		genusRow.setGenusName(row[5]);
		genusRow.setGenusAuthority(row[6]);
		genusRow.setSubGenusName(row[7]);
		genusRow.setSectionName(row[8]);
		genusRow.setSubSectionName(row[9]);
		genusRow.setSeriesName(row[10]);
		genusRow.setSubSeriesName(row[11]);
		genusRow.setNote(row[12]);
		genusRow.setCreatedDate(toDate(row[13]));
		genusRow.setCreatedById(toLong(row[14]));
		genusRow.setModifiedDate(toDate(row[15]));
		genusRow.setModifiedById(toLong(row[16]));
		genusRow.setOwnedDate(toDate(row[17]));
		genusRow.setOwnedById(toLong(row[18]));

		return genusRow;
	}

	/**
	 * Convert row to {@link SpeciesRow}.
	 *
	 * @param row in the right CSV order
	 * @return the species row
	 * @throws ParseException if date cannot be parsed
	 */
	public static SpeciesRow toSpecies(String[] row) throws ParseException {
		if (row == null || row.length == 0) {
			return null;
		}

		SpeciesRow speciesRow = new SpeciesRow();

		speciesRow.setSpeciesId(toLong(row[0]));
		speciesRow.setCurrentSpeciesId(toLong(row[1]));
		speciesRow.setNomenNumber(toLong(row[2]));
		speciesRow.setSpecificHybrid(toBoolean(row[3]));
		speciesRow.setSpeciesName(row[4]);
		speciesRow.setSpeciesAuthority(row[5]);
		speciesRow.setSubspecificHybrid(toBoolean(row[6]));
		speciesRow.setSubspeciesName(row[7]);
		speciesRow.setSubspeciesAuthority(row[8]);
		speciesRow.setVarietalHybrid(toBoolean(row[9]));
		speciesRow.setVarietyName(row[10]);
		speciesRow.setVarietyAuthority(row[11]);
		speciesRow.setSubvarietalHybrid(toBoolean(row[12]));
		speciesRow.setSubvarietyName(row[13]);
		speciesRow.setSubvarietyAuthority(row[14]);
		speciesRow.setFormaHybrid(toBoolean(row[15]));
		speciesRow.setFormaRankType(row[16]);
		speciesRow.setFormaName(row[17]);
		speciesRow.setFormaAuthority(row[18]);
		speciesRow.setGenusId(toLong(row[19]));
		speciesRow.setPrioritySite1(row[20]);
		speciesRow.setPrioritySite2(row[21]);
		speciesRow.setCurator1Id(toLong(row[22]));
		speciesRow.setCurator2Id(toLong(row[23]));
		speciesRow.setRestrictionCode(row[24]);
		speciesRow.setLifeFormCode(row[25]);
		speciesRow.setCommonFertilizationCode(row[26]);
		speciesRow.setNamePending(toBoolean(row[27]));
		speciesRow.setSynonymCode(row[28]);
		speciesRow.setVerifierId(toLong(row[29]));
		speciesRow.setNameVerifiedDate(toDate(row[30]));
		speciesRow.setName(row[31]);
		speciesRow.setNameAuthority(row[32]);
		speciesRow.setProtologue(row[33]);
		speciesRow.setNote(row[34]);
		speciesRow.setSiteNote(row[35]);
		speciesRow.setAlternateName(row[36]);
		speciesRow.setCreatedDate(toDate(row[37]));
		speciesRow.setCreatedById(row[38]);
		speciesRow.setModifiedDate(toDate(row[39]));
		speciesRow.setModifiedById(row[40]);
		speciesRow.setOwnedDate(toDate(row[41]));
		speciesRow.setOwnedById(row[42]);

		return speciesRow;
	}

	/**
	 * Convert a string in agreed format (yyyy-MM-dd HH:mm:ss) to Date.
	 *
	 * @param string the string
	 * @return the date
	 * @throws ParseException the parse exception
	 */
	public static Date toDate(String string) throws ParseException {
		if (string == null) {
			return null;
		}
		return sdf.parse(string);
	}

	/**
	 * These CSVs contain Y or N for booleans.
	 *
	 * @param string the string
	 * @return the boolean
	 */
	private static Boolean toBoolean(String string) {
		if (string == null) {
			return null;
		}
		return "Y".equals(string);
	}

	/**
	 * To long.
	 *
	 * @param string the string
	 * @return the long
	 */
	private static Long toLong(String string) {
		if (string == null) {
			return null;
		}
		return Long.parseLong(string);
	}
}
