/*
 * Copyright 2016 Global Crop Diversity Trust
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.genesys.taxonomy.checker;

import java.util.List;

import javax.annotation.Nonnull;

import org.genesys.taxonomy.checker.TaxonomyChecker.Taxon;
import org.genesys.taxonomy.gringlobal.model.IGrinSpecies;

/**
 * Defines searches possible against a database of taxonomic data.
 */
public interface TaxonomyDatabase {

	/**
	 * Does the database contain at least one record that exactly matches specified genus?.
	 *
	 * @param genus the genus
	 * @return <code>true</code> if there is at least one record
	 */
	boolean containsGenus(String genus);

	/**
	 * Return a list of genera close to the specified genus.
	 * 
	 * @param genus genus
	 * @param maxSize maximum rows to return
	 * @return list of suggestions, ordered by preference (best first), never null.
	 */
	List<String> findSimilarGenus(String genus, int maxSize);

	/**
	 * Does the database contain at least one record that exactly matches specified genus and species?.
	 *
	 * @param genus the genus
	 * @param species the species
	 * @return <code>true</code> if there is at least one record
	 */
	boolean containsSpecies(String genus, String species);

	/**
	 * Does the database contain at least one record that exactly matches specified genus and species?.
	 *
	 * @param genus the genus
	 * @param species the species
	 * @param spAuthor species authority
	 * @return <code>true</code> if there is at least one record
	 */
	boolean containsSpecies(String genus, String species, String spAuthor);

	/**
	 * Return a list of species close to the specified genus and species.
	 *
	 * @param genus valid genus
	 * @param species species
	 * @param spAuthor species authority
	 * @param maxSize maximum number of suggestions to return
	 * @return list of suggestions, ordered by preference (best first), never null.
	 */
	List<String> findSimilarSpecies(String genus, String species, String spAuthor, int maxSize);

	/**
	 * Return the species authority for the specified species within genus.
	 *
	 * @param genus the genus
	 * @param species the species
	 * @return the authority or <code>null</code>
	 */
	List<String> getSpeciesAuthority(String genus, String species);

	/**
	 * Return the species authority for the specified species within genus.
	 *
	 * @param genus the genus
	 * @param species the species
	 * @param spAuthor species authority
	 * @return the authority or <code>null</code>
	 */
	List<String> getSpeciesAuthority(String genus, String species, String spAuthor);

	/**
	 * Does the database contain at least one record that exactly matches specified genus, species and subtaxa?.
	 * 
	 * @param genus valid genus
	 * @param species valid species
	 * @param subtaxa subtaxa to check
	 * @return <code>true</code> if there is at least one record
	 */
	boolean containsSubtaxa(String genus, String species, String subtaxa);

	/**
	 * Does the database contain at least one record that exactly matches specified genus, species and subtaxa?.
	 * 
	 * @param genus valid genus
	 * @param species valid species
	 * @param spAuthor species authority
	 * @param subtaxa subtaxa to check
	 * @param subtAuthor species authority at the lowest classification level
	 * @return <code>true</code> if there is at least one record
	 */
	boolean containsSubtaxa(String genus, String species, String spAuthor, String subtaxa, String subtAuthor);

	/**
	 * Suggest a list of subtaxa
	 * 
	 * @param genus valid genus
	 * @param species valid species
	 * @param subtaxa subtaxa to check
	 * @param maxSize maximum number of suggestions to return
	 * @return list of suggestions, ordered by preference, never null
	 */
	List<String> findSimilarSubtaxa(String genus, String species, String subtaxa, int maxSize);

	/**
	 * Suggest a list of subtaxa
	 * 
	 * @param genus valid genus
	 * @param species valid species
	 * @param spAuthor species authority
	 * @param subtaxa subtaxa to check
	 * @param subtAuthor species authority at the lowest classification level
	 * @param maxSize maximum number of suggestions to return
	 * @return list of suggestions, ordered by preference, never null
	 */
	List<String> findSimilarSubtaxa(String genus, String species, String spAuthor, String subtaxa, String subtAuthor, int maxSize);

	/**
	 * Suggest a list of matches.
	 * 
	 * @param genus valid genus
	 * @param species valid species
	 * @param spAuthor species authority
	 * @param subtaxa subtaxa to check
	 * @param subtAuthor species authority at the lowest classification level
	 * @param maxSize maximum number of suggestions to return
	 * @return list of suggestions, ordered by preference, never null
	 */
	@Nonnull List<Taxon> findSimilar(String genus, String species, String spAuthor, String subtaxa, String subtAuthor, int maxSize);

	/**
	 * Return the authority for the specified subtaxa
	 *
	 * @param genus the genus
	 * @param species the species
	 * @param subtaxa subtaxa to check
	 * @return the authority or <code>null</code>
	 */
	List<String> getSubtaxaAuthority(String genus, String species, String subtaxa);

	/**
	 * Return the authority for the specified subtaxa
	 *
	 * @param genus the genus
	 * @param species the species
	 * @param spAuthor species authority
	 * @param subtaxa subtaxa to check
	 * @param subtAuthor species authority at the lowest classification level
	 * @return the authority or <code>null</code>
	 */
	List<String> getSubtaxaAuthority(String genus, String species, String spAuthor, String subtaxa, String subtAuthor);

	/**
	 * Return all subtaxa for genus and species
	 * 
	 * @param genus valid genus
	 * @param species valid species
	 * @param maxSize maximum number of rows to return
	 * @return list of species matching genus and species name
	 */
	List<IGrinSpecies> listSpecies(String genus, String species, int maxSize);

	/**
	 * Find SpeciesRows matching the definition
	 * 
	 * @param genus the genus
	 * @param species the species
	 * @param subtaxa the MCPD subtaxa
	 * @return list of matching SpeciesRows
	 */
	List<IGrinSpecies> findSpeciesRow(String genus, String species, String subtaxa);

	/**
	 * Find SpeciesRows matching the definition
	 * 
	 * @param genus the genus
	 * @param species the species
	 * @param spAuthor species authority
	 * @param subtaxa the MCPD subtaxa
	 * @param subtAuthor species authority at the lowest classification level
	 * @return list of matching SpeciesRows
	 */
	List<IGrinSpecies> findSpeciesRows(String genus, String species, String spAuthor, String subtaxa, String subtAuthor);

	/**
	 * Get exact match.
	 * 
	 * @param genus valid genus
	 * @param species valid species
	 * @param spAuthor species authority
	 * @param subtaxa subtaxa to check
	 * @param subtAuthor species authority at the lowest classification level
	 * @return One exactly matching record
	 */
	Taxon getTaxon(String genus, String species, String spAuthor, String subtaxa, String subtAuthor);

	/**
	 * @param genus      the genus
	 * @param species    the species
	 * @param spAuthor   species authority
	 * @param subtaxa    the MCPD subtaxa
	 * @param subtAuthor species authority at the lowest classification level
	 * @param maxSize    maximum rows to return
	 * @return list of taxa
	 */
	@Nonnull
	List<Taxon> findTaxa(String genus, String species, String spAuthor, String subtaxa, String subtAuthor, int maxSize);

	/**
	 * Get species by speciesId
	 * 
	 * @param speciesId duh
	 * @return SpeciesRow or null if not found
	 */
	IGrinSpecies getSpeciesRow(long speciesId);

	/**
	 * Get genus by genusId
	 * 
	 * @param genusId duh
	 * @return genus name or null
	 */
	String getGenus(long genusId);

	/**
	 * Get string similarity score.
	 *
	 * @param string1 the string 1
	 * @param string2 the string 2
	 * @return the double
	 */
	double similarityScore(String string1, String string2);


}
