/*
 * Decompiled with CFR 0.152.
 */
package org.genesys.taxonomy.checker;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.lang3.StringUtils;
import org.genesys.taxonomy.checker.BestScore;
import org.genesys.taxonomy.checker.StringSimilarity;
import org.genesys.taxonomy.checker.Suggestion;
import org.genesys.taxonomy.checker.TaxonomyDatabase;
import org.genesys.taxonomy.checker.TaxonomyException;
import org.genesys.taxonomy.gringlobal.model.SpeciesRow;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InMemoryTaxonomyDatabase
implements TaxonomyDatabase {
    public static final double NONMATCH_MARGIN = 0.8;
    private static final Logger LOG = LoggerFactory.getLogger(InMemoryTaxonomyDatabase.class);
    private static final String HYBRID_SPECIES_PREFIX = "x ";
    private static final String HYBRID_MARKER = " x ";
    private static final String[] HYBRID_MARKER_ALT = new String[]{" x ", " X "};
    private static final Pattern HYBRID_MARKER_REGEXP = Pattern.compile("\\s+[xX]\\s+");
    private Map<Long, String> genusIdLookup = new HashMap<Long, String>();
    private Map<String, List<Long>> genusIdSpecies = new HashMap<String, List<Long>>();
    private Map<Long, List<SpeciesRow>> speciesLookup = new HashMap<Long, List<SpeciesRow>>();
    private Map<Long, SpeciesRow> speciesIdLookup = new HashMap<Long, SpeciesRow>();
    private int speciesRows;

    public void registerGenus(Long genusId, String genus) {
        if (!this.genusIdSpecies.containsKey(genus)) {
            this.genusIdSpecies.put(genus, new ArrayList(1));
        }
        this.genusIdLookup.put(genusId, genus);
        this.genusIdSpecies.get(genus).add(genusId);
        this.speciesLookup.put(genusId, new ArrayList(1));
    }

    public void registerSpecies(SpeciesRow speciesRow) throws TaxonomyException {
        List<SpeciesRow> genusSpecies = this.speciesLookup.get(speciesRow.getGenusId());
        if (genusSpecies == null) {
            throw new TaxonomyException("No genus with specified genusId");
        }
        genusSpecies.add(speciesRow);
        if (this.speciesIdLookup.containsKey(speciesRow.getSpeciesId())) {
            throw new TaxonomyException("Duplicate speciesId not allowed");
        }
        this.speciesIdLookup.put(speciesRow.getSpeciesId(), speciesRow);
        ++this.speciesRows;
    }

    public String toString() {
        return "InMemory Taxonomy Database: " + this.genusIdSpecies.size() + " genera and " + this.speciesRows + " species";
    }

    @Override
    public boolean containsGenus(String genus) {
        return this.genusIdSpecies.containsKey(genus);
    }

    @Override
    public List<String> findSimilarGenus(String genus, int maxSize) {
        if (this.containsGenus(genus)) {
            LOG.trace("Database contains genus={}", (Object)genus);
            return Collections.emptyList();
        }
        BestScore bestScore = new BestScore();
        return ((Stream)this.genusIdSpecies.keySet().parallelStream().map(candidate -> new Suggestion<String>((String)candidate, this.similarityScore(genus, (String)candidate))).filter(scored -> scored.getScore() >= 0.5).sequential()).peek(scored -> bestScore.update(scored.getScore())).sorted(Comparator.comparing(Suggestion::getScore, Comparator.reverseOrder())).filter(scored -> scored.getScore() >= (bestScore.getBestScore() == 1.0 ? 0.95 : bestScore.getBestScore() * 0.8)).peek(InMemoryTaxonomyDatabase::print).map(Suggestion::getSuggestion).distinct().limit(maxSize).collect(Collectors.toList());
    }

    public static <T> void print(Suggestion<T> suggestion) {
        if (LOG.isTraceEnabled()) {
            LOG.trace("Score={} suggestion={}", (Object)suggestion.getScore(), suggestion.getSuggestion());
        }
    }

    public static void print(Object suggestion) {
        if (LOG.isTraceEnabled()) {
            LOG.trace(suggestion == null ? "NULL" : suggestion.getClass() + "=" + suggestion.toString());
        }
    }

    protected List<SpeciesRow> getAllGenusSpecies(String genus) {
        if (!this.genusIdSpecies.containsKey(genus)) {
            return Collections.emptyList();
        }
        return this.genusIdSpecies.get(genus).stream().map(genusId -> this.speciesLookup.get(genusId)).reduce(new ArrayList(1), (all, genusSpecies) -> {
            all.addAll(genusSpecies);
            return all;
        });
    }

    @Override
    public boolean containsSpecies(String genus, String species) {
        LOG.trace("Does database contain genus={} species={}", (Object)genus, (Object)species);
        if (!this.genusIdSpecies.containsKey(genus)) {
            return false;
        }
        boolean isSpecificHybrid = this.isNameSpecificHybrid(species);
        boolean isHybrid = this.isNameHybrid(species);
        if (isHybrid) {
            String[] split = HYBRID_MARKER_REGEXP.split(species);
            String speciesLeft = split[0];
            String speciesRight = split[1];
            if (LOG.isTraceEnabled()) {
                LOG.trace("Species {} is a hybrid of {} and {}", new Object[]{species, speciesLeft, speciesRight});
            }
            return this.containsSpecies(genus, speciesLeft) && this.containsSpecies(genus, speciesRight);
        }
        return this.getAllGenusSpecies(genus).stream().anyMatch(speciesRow -> {
            if (isSpecificHybrid) {
                return true == speciesRow.getSpecificHybrid() && StringUtils.equals((CharSequence)species.substring(HYBRID_SPECIES_PREFIX.length()), (CharSequence)speciesRow.getSpeciesName());
            }
            return StringUtils.equals((CharSequence)species, (CharSequence)speciesRow.getSpeciesName());
        });
    }

    @Override
    public List<String> findSimilarSpecies(String genus, String species, int maxSize) {
        LOG.debug("Searching similar species for genus={} species={}", (Object)genus, (Object)species);
        if ("Unknown".equals(genus)) {
            return Collections.emptyList();
        }
        List<Long> genusId = this.genusIdSpecies.get(genus);
        if (genusId == null) {
            throw new UnsupportedOperationException("Genus does not exist in database. Genus=" + genus);
        }
        boolean isHybrid = this.isNameHybrid(species);
        if (isHybrid) {
            String[] split = HYBRID_MARKER_REGEXP.split(species);
            String speciesLeft = split[0];
            String speciesRight = split[1];
            if (LOG.isTraceEnabled()) {
                LOG.trace("Species {} is a hybrid of {} and {}", new Object[]{species, speciesLeft, speciesRight});
            }
            List<Suggestion<String>> lefts = this.makeSuggestions(genus, speciesLeft).limit(maxSize).map(suggestion -> new Suggestion<String>(((SpeciesRow)suggestion.getSuggestion()).getSpeciesName(), suggestion.getScore())).collect(Collectors.toList());
            List<Suggestion<String>> rights = this.makeSuggestions(genus, speciesRight).limit(maxSize).map(suggestion -> new Suggestion<String>(((SpeciesRow)suggestion.getSuggestion()).getSpeciesName(), suggestion.getScore())).collect(Collectors.toList());
            if (lefts.size() == 0 && rights.size() > 0 && ((Suggestion)rights.get(0)).getScore() < 1.0) {
                lefts.add(new Suggestion<String>(speciesLeft, 0.1));
            } else if (rights.size() == 0 && lefts.size() > 0 && ((Suggestion)lefts.get(0)).getScore() < 1.0) {
                rights.add(new Suggestion<String>(speciesRight, 0.1));
            }
            if (LOG.isTraceEnabled()) {
                LOG.trace("Left for {} is {}", (Object)speciesLeft, lefts);
                LOG.trace("Right for {} is {}", (Object)speciesRight, rights);
            }
            return this.crossJoinSpecies(lefts, rights).stream().sorted(Comparator.comparing(Suggestion::getScore, Comparator.reverseOrder())).map(Suggestion::getSuggestion).distinct().limit(maxSize).collect(Collectors.toList());
        }
        return this.makeSuggestions(genus, species).map(Suggestion::getSuggestion).map(SpeciesRow::getSpeciesName).distinct().limit(maxSize).collect(Collectors.toList());
    }

    private List<Suggestion<String>> crossJoinSpecies(List<Suggestion<String>> lefts, List<Suggestion<String>> rights) {
        ArrayList<Suggestion<String>> crossJoin = new ArrayList<Suggestion<String>>();
        for (Suggestion<String> l : lefts) {
            for (Suggestion<String> r : rights) {
                Suggestion<String> j = new Suggestion<String>(l.getSuggestion().concat(HYBRID_MARKER).concat(r.getSuggestion()), l.getScore() * r.getScore());
                if (LOG.isTraceEnabled()) {
                    LOG.trace("Cross-join '{}' with '{}' result={}", new Object[]{l, r, j});
                }
                crossJoin.add(j);
            }
        }
        return crossJoin;
    }

    private Stream<Suggestion<SpeciesRow>> makeSuggestions(String genus, String species) {
        BestScore bestScore = new BestScore();
        return this.getAllGenusSpecies(genus).stream().map(speciesRow -> new Suggestion<SpeciesRow>((SpeciesRow)speciesRow, this.similarityScore(species, speciesRow.getSpeciesName()))).filter(scored -> scored.getScore() >= 0.5).peek(scored -> scored.setScore((((SpeciesRow)scored.getSuggestion()).isCurrent() ? 1.0 : 0.8) * scored.getScore())).sorted(Comparator.comparing(Suggestion::getScore, Comparator.reverseOrder())).peek(scored -> bestScore.update(scored.getScore())).filter(scored -> scored.getScore() >= (bestScore.getBestScore() == 1.0 ? 0.95 : bestScore.getBestScore() * 0.8));
    }

    @Override
    public double similarityScore(String original, String candidate) {
        return (StringSimilarity.diceCoefficientOptimized(original.toLowerCase(), candidate.toLowerCase()) + StringSimilarity.getLevenshteinCoefficient(original.toLowerCase(), candidate.toLowerCase())) / 2.0;
    }

    @Override
    public String getSpeciesAuthority(String genus, String species) {
        List<Long> genusId = this.genusIdSpecies.get(genus);
        if (genusId == null) {
            return null;
        }
        boolean isSpecificHybrid = this.isNameSpecificHybrid(species);
        return this.getAllGenusSpecies(genus).stream().filter(speciesRow -> {
            if (isSpecificHybrid) {
                return true == speciesRow.getSpecificHybrid() && StringUtils.equals((CharSequence)species.substring(HYBRID_SPECIES_PREFIX.length()), (CharSequence)speciesRow.getSpeciesName());
            }
            return StringUtils.equals((CharSequence)species, (CharSequence)speciesRow.getSpeciesName());
        }).peek(speciesRow -> LOG.trace("Species authority {}", (Object)speciesRow.getSpeciesAuthority())).findFirst().map(speciesRow -> speciesRow.getSpeciesAuthority()).orElse(null);
    }

    private boolean isNameSpecificHybrid(String name) {
        return StringUtils.startsWith((CharSequence)name, (CharSequence)HYBRID_SPECIES_PREFIX);
    }

    private boolean isNameHybrid(String name) {
        for (String opt : HYBRID_MARKER_ALT) {
            if (!StringUtils.contains((CharSequence)name, (CharSequence)opt)) continue;
            return true;
        }
        return false;
    }

    @Override
    public boolean containsSubtaxa(String genus, String species, String subtaxa) {
        LOG.trace("Does database contain genus={} species={}", (Object)genus, (Object)species);
        if (!this.genusIdSpecies.containsKey(genus)) {
            return false;
        }
        return this.getAllGenusSpecies(genus).stream().filter(speciesRow -> StringUtils.equals((CharSequence)species, (CharSequence)speciesRow.getSpeciesName())).anyMatch(speciesRow -> StringUtils.equals((CharSequence)subtaxa, (CharSequence)speciesRow.getSubtaxa()));
    }

    @Override
    public List<String> findSimilarSubtaxa(String genus, String species, String subtaxa, int maxSize) {
        LOG.debug("Searching similar subtaxa for genus={} species={} subtaxa={}", new Object[]{genus, species, subtaxa});
        List<Long> genusId = this.genusIdSpecies.get(genus);
        if (genusId == null) {
            throw new UnsupportedOperationException("Genus does not exist in database. Genus=" + genus);
        }
        BestScore bestScore = new BestScore();
        return this.getAllGenusSpecies(genus).stream().filter(speciesRow -> species.equals(speciesRow.getSpeciesName())).filter(speciesRow -> StringUtils.isNotBlank((CharSequence)speciesRow.getSubtaxa())).map(speciesRow -> new Suggestion<SpeciesRow>((SpeciesRow)speciesRow, this.similarityScore(subtaxa, speciesRow.getSubtaxa()))).filter(scored -> scored.getScore() >= 0.5).peek(scored -> scored.setScore((((SpeciesRow)scored.getSuggestion()).isCurrent() ? 1.0 : 0.8) * scored.getScore())).sorted(Comparator.comparing(Suggestion::getScore, Comparator.reverseOrder())).peek(scored -> bestScore.update(scored.getScore())).filter(scored -> scored.getScore() >= (bestScore.getBestScore() == 1.0 ? 0.95 : bestScore.getBestScore() * 0.8)).map(Suggestion::getSuggestion).map(SpeciesRow::getSubtaxa).distinct().limit(maxSize).collect(Collectors.toList());
    }

    @Override
    public String getSubtaxaAuthority(String genus, String species, String subtaxa) {
        List<Long> genusId = this.genusIdSpecies.get(genus);
        if (genusId == null) {
            return null;
        }
        return this.getAllGenusSpecies(genus).stream().filter(speciesRow -> StringUtils.equals((CharSequence)species, (CharSequence)speciesRow.getSpeciesName())).filter(speciesRow -> StringUtils.equals((CharSequence)subtaxa, (CharSequence)speciesRow.getSubtaxa())).peek(speciesRow -> LOG.trace("Subtaxa authority {}", (Object)speciesRow.getSubtaxaAuthority())).findFirst().map(speciesRow -> speciesRow.getSubtaxaAuthority()).orElse(null);
    }

    @Override
    public List<SpeciesRow> listSpecies(String genus, String species, int maxSize) {
        return this.getAllGenusSpecies(genus).stream().filter(speciesRow -> StringUtils.equals((CharSequence)species, (CharSequence)speciesRow.getSpeciesName())).limit(maxSize).collect(Collectors.toList());
    }

    @Override
    public List<SpeciesRow> findSpeciesRow(String genus, String species, String subtaxa) {
        boolean isSpecificHybrid = this.isNameSpecificHybrid(species);
        return this.getAllGenusSpecies(genus).stream().filter(speciesRow -> {
            if (isSpecificHybrid) {
                return true == speciesRow.getSpecificHybrid() && StringUtils.equals((CharSequence)species.substring(HYBRID_SPECIES_PREFIX.length()), (CharSequence)speciesRow.getSpeciesName());
            }
            return StringUtils.equals((CharSequence)species, (CharSequence)speciesRow.getSpeciesName());
        }).filter(speciesRow -> StringUtils.equals((CharSequence)subtaxa, (CharSequence)speciesRow.getSubtaxa())).collect(Collectors.toList());
    }

    @Override
    public SpeciesRow getSpeciesRow(long speciesId) {
        return this.speciesIdLookup.get(speciesId);
    }

    @Override
    public String getGenus(long genusId) {
        return this.genusIdLookup.get(genusId);
    }
}

