/*
 * Copyright 2017 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.taxonomy.checker;

import java.util.List;
import java.util.concurrent.ExecutionException;

import org.genesys.taxonomy.gringlobal.model.IGrinSpecies;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

/**
 * Guava cache supported in-memory taxonomy database.
 */
public class CachingInMemoryTaxonomyDatabase extends InMemoryTaxonomyDatabase {
	// private final static Logger LOG =
	// LoggerFactory.getLogger(CachingInMemoryTaxonomyDatabase.class);

	/** The cache genus species. */
	private final Cache<String, List<IGrinSpecies>> cacheGenusSpecies = CacheBuilder.newBuilder().maximumSize(100).build();

	/** The cache genus suggestions. */
	private final Cache<String, List<String>> cacheGenusSuggestions = CacheBuilder.newBuilder().maximumSize(100).build();

	/** The cache species suggestions. */
	private final Cache<String, List<String>> cacheSpeciesSuggestions = CacheBuilder.newBuilder().maximumSize(100).build();

	/**
	 * Gets the all genus species.
	 *
	 * @param genus the genus
	 * @return the all genus species
	 */
	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.taxonomy.checker.InMemoryTaxonomyDatabase#getAllGenusSpecies(java
	 * .lang.String)
	 */
	@Override
	protected List<IGrinSpecies> getAllGenusSpecies(final String genus) {
		try {
			return cacheGenusSpecies.get(genus, () -> CachingInMemoryTaxonomyDatabase.super.getAllGenusSpecies(genus));
		} catch (final ExecutionException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Find similar genus.
	 *
	 * @param genus the genus
	 * @param maxSize the max size
	 * @return the list
	 */
	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.taxonomy.checker.InMemoryTaxonomyDatabase#findSimilarGenus(java.
	 * lang.String, int)
	 */
	@Override
	public List<String> findSimilarGenus(final String genus, final int maxSize) {
		try {
			return cacheGenusSuggestions.get(genus + "-" + maxSize, () -> CachingInMemoryTaxonomyDatabase.super.findSimilarGenus(genus, maxSize));
		} catch (final ExecutionException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Find similar species.
	 *
	 * @param genus the genus
	 * @param species the species
	 * @param maxSize the max size
	 * @return the list
	 */
	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.taxonomy.checker.InMemoryTaxonomyDatabase#findSimilarSpecies(java
	 * .lang.String, java.lang.String, int)
	 */
	@Override
	public List<String> findSimilarSpecies(final String genus, final String species, final int maxSize) {
		try {
			return cacheSpeciesSuggestions.get(genus + "-" + species + "-" + maxSize, () -> CachingInMemoryTaxonomyDatabase.super.findSimilarSpecies(genus, species, maxSize));
		} catch (final ExecutionException e) {
			throw new RuntimeException(e);
		}
	}
}
