/*
 * Copyright 2016 Global Crop Diversity Trust
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.genesys.taxonomy.checker;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Suggest fixes to scientific names based on an in-memory list of valid taxonomies.
 */
public class TaxonomyChecker {

	private static final String UNKNOWN_GENUS = "Unknown";
	private static final String[] UNKNOWN_GENUS_ALT = { UNKNOWN_GENUS, "Unidentified" };

	private static final String UNKNOWN_SPECIES = "sp.";
	private static final String[] UNKNOWN_SPECIES_ALT = { UNKNOWN_SPECIES, "spp." };

	/** The Constant LOG. */
	private final static Logger LOG = LoggerFactory.getLogger(TaxonomyChecker.class);

	/** The database. */
	private TaxonomyDatabase database;

	/**
	 * Sets the taxonomy database.
	 *
	 * @param database the new taxonomy database
	 */
	public void setTaxonomyDatabase(TaxonomyDatabase database) {
		this.database = database;
	}

	/**
	 * Find suggestions for GENUS.
	 *
	 * @param genus the genus
	 * @param maxSize the max size
	 * @return suggested fixes for genus or empty list when genus is fine or when there are no suggestions.
	 */
	public List<String> suggestGenus(String genus, int maxSize) {
		if (StringUtils.isBlank(genus) || ArrayUtils.contains(UNKNOWN_GENUS_ALT, genus)) {
			return Arrays.asList(UNKNOWN_GENUS);
		} else if (database.containsGenus(genus)) {
			LOG.trace("Database contains genus={}", genus);
			return Arrays.asList(genus);
		} else {
			LOG.debug("Database does not contain genus={}", genus);
		}
		return database.findSimilarGenus(genus, maxSize);
	}

	/**
	 * Find suggestions for SPECIES.
	 *
	 * @param genus the genus
	 * @param species the species
	 * @param maxSize maximum number of suggestions to return
	 * @return suggested fixes for genus or empty list if species is fine or when there are no suggestions.
	 */
	public List<String> suggestSpecies(String genus, String species, int maxSize) {
		LOG.debug("Suggesting species for genus={} species={}", genus, species);

		if (StringUtils.isBlank(species)) {
			return Arrays.asList(UNKNOWN_SPECIES);
		}
		if (ArrayUtils.contains(UNKNOWN_SPECIES_ALT, species)) {
			return List.of(species);
		}
		if (ArrayUtils.contains(UNKNOWN_GENUS_ALT, genus)) {
			return Arrays.asList(UNKNOWN_SPECIES);
		}

		if (database.containsSpecies(genus, species)) {
			return Arrays.asList(species);
		}

		if (database.containsGenus(genus)) {
			// Genus exists
			LOG.debug("Database contains genus={}", genus);
			return database.findSimilarSpecies(genus, species, maxSize);
		} else {
			LOG.debug("Database does not contain genus={}", genus);
		}

		// Genus not listed in the database, go through suggestions
		List<String> suggestions = new ArrayList<>(0);
		for (String suggestedGenus : suggestGenus(genus, 2)) {
			LOG.debug("Suggesting with suggested genus={}", suggestedGenus);
			suggestions.addAll(database.findSimilarSpecies(suggestedGenus, species, 2).stream()
					.map(suggestion -> suggestedGenus.equalsIgnoreCase(genus) ? suggestion : "(" + suggestedGenus + ") " + suggestion).collect(Collectors.toList()));
		}
		return suggestions;
	}

	/**
	 * Return species authority for the genus + species.
	 *
	 * @param genus the genus
	 * @param species the species
	 * @return species authority if there's one matching record in the database
	 */
	public String getSpeciesAuthority(String genus, String species) {
		if (!database.containsSpecies(genus, species)) {
			return null;
		}

		return database.getSpeciesAuthority(genus, species);
	}

	/**
	 * Find suggestions for SUBTAXA.
	 *
	 * @param genus must be valid genus in the database
	 * @param species species must be valid species within genus
	 * @param subtaxa current subtaxa, must not be null or blank
	 * @param maxSize maximum number of suggestions to return
	 * @return suggested fixes for subtaxa or empty list if there are no suggestions.
	 */
	public List<String> suggestSubtaxa(String genus, String species, String subtaxa, int maxSize) {
		if (StringUtils.isBlank(subtaxa) || !database.containsSpecies(genus, species)) {
			return Collections.emptyList();
		}
		if (database.containsSubtaxa(genus, species, subtaxa)) {
			return Arrays.asList(subtaxa);
		}
		return database.findSimilarSubtaxa(genus, species, subtaxa, maxSize);
	}

	/**
	 * Return authority for the genus + species + subtaxa
	 *
	 * @param genus the genus
	 * @param species the species
	 * @param subtaxa subtaxa
	 * @return species authority if there's one matching record in the database
	 */
	public String getSubtaxaAuthority(String genus, String species, String subtaxa) {
		if (!database.containsSubtaxa(genus, species, subtaxa)) {
			return null;
		}

		return database.getSubtaxaAuthority(genus, species, subtaxa);
	}
	
	/**
	 * Get string similarity score.
	 *
	 * @param string1 string A
	 * @param string2 string B
	 * @return the double
	 */
	public double similarityScore(final String string1, final String string2) {
		return database.similarityScore(string1, string2);
	}

}
