/**
 * Copyright 2014 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/

package org.genesys2.transifex.client;

import java.io.File;
import java.io.IOException;
import java.util.Locale;

/**
 * Transifex client.
 *
 * @author matijaobreza
 */
public interface TransifexService {

	/**
	 * Checks if resource exists on Transifex
	 *
	 * @param slug the resource slug
	 * @return true, if successful
	 */
	boolean resourceExists(String slug);

	/**
	 * Delete resource.
	 *
	 * @param slug the resource slug
	 * @return true, if successful
	 * @throws TransifexException the transifex exception
	 */
	boolean deleteResource(String slug) throws TransifexException;

	/**
	 * Gets the translated resource.
	 *
	 * @param slug the resource slug
	 * @param locale the locale
	 * @return the translated resource
	 * @throws TransifexException the transifex exception
	 */
	String getTranslatedResource(String slug, Locale locale) throws TransifexException;

	/**
	 * Create a new XHTML resource. Fails if resource exists.
	 *
	 * @param slug the slug
	 * @param title the title
	 * @param content the content
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	void createXhtmlResource(String slug, String title, String content) throws IOException;

	/**
	 * Create or update XHTML resource.
	 *
	 * @param slug the slug
	 * @param title the title
	 * @param content the content
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	void updateXhtmlResource(String slug, String title, String content) throws IOException;

	/**
	 * Create or update Java XML Properties resource.
	 *
	 * @param slug the slug
	 * @param title the title
	 * @param file the file
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	void updatePropertiesXmlResource(String slug, String title, File file) throws IOException;

	/**
	 * Download translated resource.
	 *
	 * @param slug the slug
	 * @param locale the locale
	 * @return the string
	 * @throws TransifexException the transifex exception
	 */
	String downloadTranslatedResource(String slug, Locale locale) throws TransifexException;

	/**
	 * Ping service to test credentials
	 * 
	 * @throws TransifexException the transifex exception
	 */
	void testCredentials() throws TransifexException;

}
