/**
 * Copyright 2015 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/

package org.genesys.worldclim.grid.generic;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

public class GenericGridDataReader implements Closeable {
	private InputStream bil;
	private Header header;

	private byte[] readBuffer;
	private ByteBuffer byteBuffer;

	public GenericGridDataReader(InputStream bil, Header header) {
		this.bil = bil;
		this.header = header;

		// Read buffer
		this.readBuffer = new byte[header.getBandRowBytes()];

		// Conversion buffer
		this.byteBuffer = ByteBuffer.allocate(header.getBandRowBytes());
		this.byteBuffer.order(ByteOrder.LITTLE_ENDIAN);
	}

	public void close() throws IOException {
		this.byteBuffer.clear();
		this.byteBuffer = null;

		this.readBuffer = null;
		this.bil.close();
	}

	public Short[] readBandData() throws IOException {
		// System.err.println("expecting " + header.getColumns()
		// + " results in band");

		// Result band
		Short[] band = new Short[header.getColumns()];

		// Read from stream
		// FIXME this looks funny! Two buffers!
		byteBuffer.clear();
		bil.read(readBuffer);
		byteBuffer.put(readBuffer);

		for (int i = 0; i < band.length; i++) {
			band[i] = byteBuffer.getShort(i * 2);
			// System.err.println(i + " = " + band[i]);

			// handle no-data-values
			if ((int) band[i] == header.getNoDataValue()) {
				// System.err.println("Is null!");
				band[i] = null;
			}
		}

		// System.err.println("Skipping " + header.getBandGapBytes());
		// Skip gap bytes!
		bil.skip(header.getBandGapBytes());

		return band;
	}
}
