/**
 * Copyright 2015 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/

package org.genesys.worldclim.grid.generic;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;

import org.apache.commons.io.IOUtils;
import org.genesys.worldclim.ZipUtils;

public class GenericGridZipFile implements Closeable {

	private ZipFile zipFile;
	private Set<String> headers;

	public GenericGridZipFile(ZipFile zipFile, Set<String> headers) {
		this.zipFile = zipFile;
		this.headers = headers;
	}

	public GenericGridZipFile(ZipFile zipFile) {
		this.zipFile = zipFile;
		this.headers = ZipUtils.listHeaders(zipFile);
	}

	public static GenericGridZipFile open(File file) throws ZipException,
			IOException {
		ZipFile zipFile = new ZipFile(file);
		Set<String> headers = ZipUtils.listHeaders(zipFile);
		return new GenericGridZipFile(zipFile, headers);
	}

	public void close() throws IOException {
		this.zipFile.close();
		this.zipFile = null;
		this.headers = null;
	}

	public Set<String> getHeaderFileNames() {
		return headers;
	}

	/**
	 * Read generic grid header data from zip entry
	 * 
	 * @param headerFileName
	 * @return
	 * @throws IOException
	 */
	public Header readHeader(String headerFileName) throws IOException {

		ZipEntry headerEntry = zipFile.getEntry(headerFileName);
		if (headerEntry == null) {
			return null;
		}

		GenericGridHeaderReader gghr = null;
		try {
			gghr = new GenericGridHeaderReader(
					zipFile.getInputStream(headerEntry));
			Header header = gghr.readHeader();
			header.setName(headerFileName.substring(0,
					headerFileName.length() - 4));
			return header;
		} finally {
			gghr.close();
		}
	}

	public GenericGridDataReader getReader(Header header) throws IOException {
		String dataFileName = header.getName() + ".bil";
		ZipEntry headerEntry = zipFile.getEntry(dataFileName);
		if (headerEntry == null) {
			throw new IOException("Data file not found " + dataFileName);
		}

		return new GenericGridDataReader(zipFile.getInputStream(headerEntry),
				header);
	}

	/**
	 * Unzip the zip archive to destination folder. Destination folder must
	 * exist and be a directory.
	 * 
	 * @param zipArchive archive to unzip
	 * @param destination
	 *            target directory
	 * @return
	 * @throws IOException
	 */
	public static List<File> unzip(File zipArchive, File destination) throws IOException {
		if (!destination.isDirectory()) {
			throw new IOException("Destination must be a directory");
		}
		if (!destination.canWrite()) {
			throw new IOException("Cannot write to destination directory");
		}

		final ZipFile zipFile=new ZipFile(zipArchive);
		final ArrayList<File> files = new ArrayList<File>();

		Enumeration<? extends ZipEntry> zipEntries = zipFile.entries();
		while (zipEntries.hasMoreElements()) {
			ZipEntry zipEntry = zipEntries.nextElement();
			String name = zipEntry.getName();
			// TODO handle sub-directories
			File destFile = new File(destination, name);

			BufferedOutputStream fos = null;
			BufferedInputStream fis = null;

			try {
				fos = new BufferedOutputStream(new FileOutputStream(destFile));
				fis = new BufferedInputStream(zipFile.getInputStream(zipEntry));
				IOUtils.copy(fis, fos);
				fos.flush();
			} finally {
				IOUtils.closeQuietly(fis);
				IOUtils.closeQuietly(fos);
			}

			files.add(destFile);
		}
		
		zipFile.close();

		return files;
	}
}
