/**
 * Copyright 2015 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/

package org.genesys.worldclim.grid.generic;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Map;

/**
 * Representation of generic grid .HDR file
 * 
 * @author mobreza
 */
public class Header {
	public enum DataType {
		I
	}

	public enum Layout {
		BIL
	}

	private String name;

	/**
	 * Byte order
	 */
	private DataType byteOrder;
	/**
	 * Layout
	 */
	private Layout layout;
	/**
	 * Number of rows
	 */
	private int rows;
	/**
	 * Number of columns
	 */
	private int columns;
	/**
	 * Number of bands
	 */
	private int bands;
	/**
	 * Number of bits per value
	 */
	private int bits;
	/**
	 * Size of one band in one row in bytes
	 */
	private int bandRowBytes;
	/**
	 * Size of one row in bytes
	 */
	private int totalRowBytes;
	/**
	 * Band padding bytes in row
	 */
	private int bandGapBytes = 0;
	/**
	 * Value used to represent missing data (NULL)
	 */
	private int noDataValue = -9999;
	/**
	 * X (longitude) of upper-left block (1st value)
	 */
	private double ulXMap;
	/**
	 * Y (latitude) of upper-left block (1st value)
	 */
	private double ulYMap;
	/**
	 * Height of block in degrees
	 */
	private double xDim;
	/**
	 * Width of block in degrees
	 */
	private double yDim;
	/**
	 * Additional attributes of the dataset
	 */
	private Map<String, String> attributes;

	public DataType getByteOrder() {
		return byteOrder;
	}

	public void setByteOrder(DataType byteOrder) {
		this.byteOrder = byteOrder;
	}

	public Layout getLayout() {
		return layout;
	}

	public void setLayout(Layout layout) {
		this.layout = layout;
	}

	public int getRows() {
		return rows;
	}

	public void setRows(int rows) {
		this.rows = rows;
	}

	public int getColumns() {
		return columns;
	}

	public void setColumns(int columns) {
		this.columns = columns;
	}

	public int getBands() {
		return bands;
	}

	public void setBands(int bands) {
		this.bands = bands;
	}

	public int getBits() {
		return bits;
	}

	public void setBits(int bits) {
		this.bits = bits;
	}

	public int getBandRowBytes() {
		return bandRowBytes;
	}

	public void setBandRowBytes(int bandRowBytes) {
		this.bandRowBytes = bandRowBytes;
	}

	public int getTotalRowBytes() {
		return totalRowBytes;
	}

	public void setTotalRowBytes(int totalRowBytes) {
		this.totalRowBytes = totalRowBytes;
	}

	public int getBandGapBytes() {
		return bandGapBytes;
	}

	public void setBandGapBytes(int bandGapBytes) {
		this.bandGapBytes = bandGapBytes;
	}

	public int getNoDataValue() {
		return noDataValue;
	}

	public void setNoDataValue(int noDataValue) {
		this.noDataValue = noDataValue;
	}

	public double getUlXMap() {
		return ulXMap;
	}

	public void setUlXMap(double ulXMap) {
		this.ulXMap = ulXMap;
	}

	public double getUlYMap() {
		return ulYMap;
	}

	public void setUlYMap(double ulYMap) {
		this.ulYMap = ulYMap;
	}

	public double getxDim() {
		return xDim;
	}

	public void setxDim(double xDim) {
		this.xDim = xDim;
	}

	public double getyDim() {
		return yDim;
	}

	public void setyDim(double yDim) {
		this.yDim = yDim;
	}

	public Map<String, String> getAttributes() {
		return attributes;
	}

	public void setAttributes(Map<String, String> attributes) {
		this.attributes = attributes;
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public ByteBuffer createByteBuffer() {
		ByteBuffer byteBuffer = ByteBuffer.allocate(getBits() / 8);
		byteBuffer.order(ByteOrder.LITTLE_ENDIAN);
		return byteBuffer;
	}
}
