/**
 * Copyright 2015 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/

package org.genesys.worldclim.grid.generic;

import java.io.BufferedReader;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.HashMap;

/**
 * Reads {@link Header} from an input stream
 * 
 * @author mobreza
 *
 */
public class GenericGridHeaderReader implements Closeable {
	private InputStream source;

	public GenericGridHeaderReader(InputStream is) {
		this.source = is;
	}

	public void close() throws IOException {
		this.source.close();
	}

	public Header readHeader() throws IOException {
		Header header = new Header();
		BufferedReader br = new BufferedReader(new InputStreamReader(
				this.source));

		String line = null;
		HashMap<String, String> attrs = null;

		boolean foundAttrs = false;
		while ((line = br.readLine()) != null) {
			if (line.length() == 0) {
				if (!foundAttrs) {
					foundAttrs = true;
					header.setAttributes(attrs = new HashMap<String, String>());
				}
				// Skip blank lines
				continue;
			}

			String[] l = line.split("\\s+", 2);
			if (foundAttrs) {
				attrs.put(l[0], l[1]);
			} else {
				setHeader(header, l[0], l[1]);
			}
		}

		return header;
	}

	private void setHeader(Header header, String key, String value)
			throws IOException {
		if ("BYTEORDER".equals(key)) {
			// TODO
		} else if ("LAYOUT".equals(key)) {
			// TODO
		} else if ("NROWS".equals(key)) {
			header.setRows(Integer.parseInt(value));
		} else if ("NCOLS".equals(key)) {
			header.setColumns(Integer.parseInt(value));
		} else if ("NBANDS".equals(key)) {
			header.setBands(Integer.parseInt(value));
		} else if ("NBITS".equals(key)) {
			header.setBits(Integer.parseInt(value));
		} else if ("BANDROWBYTES".equals(key)) {
			header.setBandRowBytes(Integer.parseInt(value));
		} else if ("TOTALROWBYTES".equals(key)) {
			header.setTotalRowBytes(Integer.parseInt(value));
		} else if ("BANDGAPBYTES".equals(key)) {
			header.setBandGapBytes(Integer.parseInt(value));
		} else if ("NODATA".equals(key)) {
			header.setNoDataValue(Integer.parseInt(value));
		} else if ("ULXMAP".equals(key)) {
			header.setUlXMap(Double.parseDouble(value));
		} else if ("ULYMAP".equals(key)) {
			header.setUlYMap(Double.parseDouble(value));
		} else if ("XDIM".equals(key)) {
			header.setxDim(Double.parseDouble(value));
		} else if ("YDIM".equals(key)) {
			header.setyDim(Double.parseDouble(value));
		} else {
			throw new IOException("Invalid header " + key);
		}
	}
}
