package org.getshaka.shaka

import scala.collection.mutable.HashSet
import scala.scalajs.js
import org.scalajs.dom.Element
import org.scalajs.dom.HTMLElement

/** Given a parent Element and Binding, returns a function from the State value
  * V to Unit.
  */
type NodeBindingBuilder[V] = (Element, Binding[?]) ?=> V => Unit

/** Given a parent Binding, returns a function from the State value V to Unit
  */
type PropBindingBuilder[V] = Binding[?] ?=> V => Unit

// will be more efficient than the 2 anon functions generated by `type`
// https://github.com/lampepfl/dotty/issues/11829
//trait NodeBindingBuilder[V]:
//  def build(v: V)(using Element, Binding[?]): Unit
// .. same for PropBindingBuilder

/** A State that can be used to bind Nodes and their properties. You can get the
  * current value with myState.value, and update with myState.value = newValue.
  * These two methods are protected, so that state update logic is encapsulated
  * within the object itself. This provides the same benefit as Redux's action -
  * reducer - store trifecta. <br>
  * @param initialValue
  *   The State's initial value, discarded if the StorageManager can fetch a
  *   value.
  * @param storageManager
  *   How the State should be persisted, such as in LocalStorage. By default,
  *   the State has no storage.
  * @tparam V
  *   the value.
  */
abstract class State[V](
    initialValue: => V,
    storageManager: StorageManager[V] = NoStorage[V]()
):
  private var _value = storageManager.fetch.getOrElse(initialValue)

  // using the javascript Set since it has nice properties..
  // ordered & safe to delete while iterating
  private val bindings = js.Set.empty[Binding[V]]

  /** Returns this State's value
    */
  protected def value: V = _value

  /** Sets this State's new value. All depdendent Bindings are recomputed.
    */
  protected def setValue(newValue: V): Unit =
    _value = newValue
    storageManager.store(newValue)
    for b <- bindings do b.onChange(newValue)

  protected def setValue(oldToNew: V => V): Unit =
    setValue(oldToNew(_value))

  /** Construct a Binding for DOM Nodes
    */
  def bind(
      builder: NodeBindingBuilder[V]
  )(using pe: Element, pb: Binding[?]): Unit =
    val b = NodeBinding[V](this, pe, builder)
    pb.addChildBinding(b)
    bindings += b
    b.onChange(_value)

  /** Construct a lighter-weight Binding for CSS/JS Props.
    */
  def bindProps(builder: PropBindingBuilder[V])(using pb: Binding[?]): Unit =
    val b = PropBinding[V](this, builder)
    pb.addChildBinding(b)
    bindings += b
    b.onChange(_value)

  /** Removes a Binding from this State.
    */
  def removeBinding(b: Binding[V]): Unit =
    bindings -= b
