/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package com.sun.enterprise.connectors;

import com.sun.appserv.connectors.internal.api.ConnectorRuntimeException;
import org.glassfish.api.admin.config.Property;
import org.glassfish.api.naming.GlassfishNamingManager;
import org.jvnet.hk2.annotations.Service;
import org.jvnet.hk2.annotations.Scoped;
import org.jvnet.hk2.component.PerLookup;
import com.sun.enterprise.config.serverbeans.ResourceAdapterConfig;
import com.sun.enterprise.connectors.util.ConnectorDDTransformUtils;
import com.sun.enterprise.connectors.util.SetMethodAction;
import com.sun.enterprise.deployment.ConnectorDescriptor;
import com.sun.enterprise.deployment.EnvironmentProperty;
import com.sun.enterprise.deployment.AdminObject;
import com.sun.enterprise.util.i18n.StringManager;
import com.sun.enterprise.resource.beans.AdministeredObjectResource;
import com.sun.logging.LogDomains;

import javax.resource.ResourceException;
import javax.resource.spi.*;
import javax.naming.Reference;
import javax.naming.NamingException;
import java.util.*;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * This class represents a live outbound resource adapter (1.5 compliant) i.e.
 * <p/>
 * A resource adapter is considered active after start()
 * and before stop() is called.
 *
 * @author Binod P G, Sivakumar Thyagarajan
 */
@Service
@Scoped(PerLookup.class)
public class ActiveOutboundResourceAdapter extends ActiveResourceAdapterImpl {

    //TODO V3 need to expose 1.5 related lifecycle methods via a contract so that ActiveJMSRA can utilize them
    protected ResourceAdapter resourceadapter_; //runtime instance

    protected String moduleName_;

    protected static final Logger _logger = LogDomains.getLogger(ActiveOutboundResourceAdapter.class, LogDomains.RSR_LOGGER);

    private StringManager localStrings =
            StringManager.getManager(ActiveOutboundResourceAdapter.class);

    protected BootstrapContext bootStrapContextImpl;

    public ActiveOutboundResourceAdapter() {
    }

    /**
     * Creates an active inbound resource adapter. Sets all RA java bean
     * properties and issues a start.
     *
     * @param ra         <code>ResourceAdapter<code> java bean.
     * @param desc       <code>ConnectorDescriptor</code> object.
     * @param moduleName Resource adapter module name.
     * @param jcl        <code>ClassLoader</code> instance.
     * @throws ConnectorRuntimeException If there is a failure in loading
     *                                   or starting the resource adapter.
     */
    public void init(
            ResourceAdapter ra, ConnectorDescriptor desc, String moduleName,
            ClassLoader jcl) throws ConnectorRuntimeException {
        super.init(ra, desc, moduleName, jcl);
        this.resourceadapter_ = ra;
        this.moduleName_ = moduleName;
        if (resourceadapter_ != null) {
            try {
                loadRAConfiguration();
                ConnectorRegistry registry = ConnectorRegistry.getInstance();
                String poolId = null;
                ResourceAdapterConfig raConfig = registry.getResourceAdapterConfig(moduleName_);
                if (raConfig != null) {
                    poolId = raConfig.getThreadPoolIds();
                }
                this.bootStrapContextImpl = new BootstrapContextImpl(poolId, moduleName_);

                startResourceAdapter(bootStrapContextImpl);

                //TODO V3 setup monitoring

            } catch (ResourceAdapterInternalException ex) {
                _logger.log(Level.SEVERE, "rardeployment.start_failed", ex);
                String i18nMsg = localStrings.getString("rardeployment.start_failed", ex.getMessage());
                ConnectorRuntimeException cre = new ConnectorRuntimeException(i18nMsg);
                cre.initCause(ex);
                throw cre;
            } catch (Throwable t) {
                _logger.log(Level.SEVERE, "rardeployment.start_failed", t);
                t.printStackTrace();
                String i18nMsg = localStrings.getString("rardeployment.start_failed", t.getMessage());
                ConnectorRuntimeException cre = new ConnectorRuntimeException(i18nMsg);
                if (t.getCause() != null) {
                    cre.initCause(t.getCause());
                } else {
                    cre.initCause(t);
                }
                throw cre;
            }
        }
    }

    //TODO V3 let ActiveJMSRA override
    protected void startResourceAdapter(BootstrapContext bootstrapContext) throws ResourceAdapterInternalException {
        resourceadapter_.start(bootstrapContext);
    }

    /**
     * {@inheritDoc}
     */
    public boolean handles(ConnectorDescriptor cd) {
        boolean adminObjectsDefined = false;
        Set adminObjects = cd.getAdminObjects();
        if (adminObjects != null && adminObjects.size() > 0) {
            adminObjectsDefined = true;
        }

        return !cd.getInBoundDefined() && (cd.getOutBoundDefined() || adminObjectsDefined) &&
                !("".equals(cd.getResourceAdapterClass()));
    }


    /**
     * Retrieves the resource adapter java bean.
     *
     * @return <code>ResourceAdapter</code>
     */
    public ResourceAdapter getResourceAdapter() {
        return this.resourceadapter_;
    }

    /**
     * Does the necessary initial setup. Creates the default pool and
     * resource.
     *
     * @throws ConnectorRuntimeException If there is a failure
     */
    public void setup() throws ConnectorRuntimeException {
        //TODO V3 need for this check ?
        if (connectionDefs_ == null || connectionDefs_.length == 0) {
            return;
        }
        super.setup();
    }

    /**
     * Destroys default pools and resources. Stops the Resource adapter
     * java bean.
     */
    public void destroy() {
        //it is possible that a 1.5 ra may not have connection-definition at all
        if ((connectionDefs_ != null) && (connectionDefs_.length != 0)) {
            super.destroy();
        }
        stopResourceAdapter();
    }

    private void stopResourceAdapter() {
        if (resourceadapter_ != null) {
            try {
                _logger.fine("Calling Resource Adapter stop" + this.getModuleName());
                resourceadapter_.stop();
                _logger.fine("Resource Adapter stop call of " + this.getModuleName() + "returned successfully");
                _logger.log(Level.FINE, "rar_stop_call_successful");
            } catch (Throwable t) {
                _logger.log(Level.SEVERE, "rardeployment.stop_warning", t);
            } finally {
                //not needed when there is no ResourceAdapter instance (implementation)
                removeProxiesFromRegistry(moduleName_);
            }
        }
    }


    /**
     * Remove all the proxy objects (Work-Manager) from connector registry
     *
     * @param moduleName_ resource-adapter name
     */
    private void removeProxiesFromRegistry(String moduleName_) {
        ConnectorRuntime.getRuntime().removeWorkManagerProxy(moduleName_);
    }


    /**
     * Creates an instance of <code>ManagedConnectionFactory</code>
     * object using the connection pool properties. Also set the
     * <code>ResourceAdapterAssociation</code>
     *
     * @param pool <code>ConnectorConnectionPool</code> properties.
     * @param jcl  <code>ClassLoader</code>
     */
    public ManagedConnectionFactory createManagedConnectionFactory(
            ConnectorConnectionPool pool, ClassLoader jcl) {
        ManagedConnectionFactory mcf;
        mcf = super.createManagedConnectionFactory(pool, jcl);

        if (mcf instanceof ResourceAdapterAssociation) {
            try {
                ((ResourceAdapterAssociation) mcf).setResourceAdapter(this.resourceadapter_);
            } catch (ResourceException ex) {
                _logger.log(Level.SEVERE, "rardeployment.assoc_failed", ex);
            }
        }
        return mcf;
    }

    /**
     * Loads RA javabean. This method is protected, so that any system
     * resource adapter can have specific configuration done during the
     * loading.
     *
     * @throws ConnectorRuntimeException if there is a failure.
     */
    protected void loadRAConfiguration() throws ConnectorRuntimeException {
        try {
            Set mergedProps;
            ConnectorRegistry registry = ConnectorRegistry.getInstance();
            ResourceAdapterConfig raConfig = registry.getResourceAdapterConfig(moduleName_);
            List<Property> raConfigProps = new ArrayList<Property>();
            mergedProps = mergeRAConfiguration(raConfig, raConfigProps);
            logMergedProperties(mergedProps);

            SetMethodAction setMethodAction = new SetMethodAction(this.resourceadapter_, mergedProps);
            setMethodAction.run();
        } catch (Exception e) {
            String i18nMsg = localStrings.getString("ccp_adm.wrong_params_for_create", e.getMessage());
            ConnectorRuntimeException cre = new ConnectorRuntimeException(i18nMsg);
            cre.initCause(e);
            throw cre;
        }
    }

    protected Set mergeRAConfiguration(ResourceAdapterConfig raConfig, List<Property> raConfigProps) {
        Set mergedProps;
        if (raConfig != null) {
            raConfigProps = raConfig.getProperty();
        }
        //TODO V3 handle JMS RA Hack
        mergedProps = ConnectorDDTransformUtils.mergeProps(raConfigProps, getDescriptor().getConfigProperties());
        return mergedProps;
    }

    private void logMergedProperties(Set mergedProps) {
        if (_logger.isLoggable(Level.FINE)) {
            _logger.fine("Passing in the following properties " +
                    "before calling RA.start of " + this.moduleName_);
            StringBuffer b = new StringBuffer();

            for (Iterator iter = mergedProps.iterator(); iter.hasNext();) {
                EnvironmentProperty element = (EnvironmentProperty) iter.next();
                b.append("\nName: " + element.getName()
                        + " Value: " + element.getValue());
            }
            _logger.fine(b.toString());
        }
    }

    public BootstrapContext getBootStrapContext() {
        return this.bootStrapContextImpl;
    }

    /**
     * Creates an admin object.
     *
     * @param appName         Name of application, in case of embedded rar.
     * @param connectorName   Module name of the resource adapter.
     * @param jndiName        JNDI name to be registered.
     * @param adminObjectType Interface name of the admin object.
     * @param props           <code>Properties</code> object containing name/value
     *                        pairs of properties.
     */
    public void addAdminObject(
            String appName,
            String connectorName,
            String jndiName,
            String adminObjectType,
            Properties props)
            throws ConnectorRuntimeException {
        if (props == null) {
            // empty properties
            props = new Properties();
        }

        ConnectorRegistry registry = ConnectorRegistry.getInstance();

        ConnectorDescriptor desc = registry.getDescriptor(connectorName);
        AdminObject aoDesc =
                desc.getAdminObjectByType(adminObjectType);

        AdministeredObjectResource aor = new AdministeredObjectResource(jndiName);
        aor.initialize(aoDesc);
        aor.setResourceAdapter(connectorName);

        Object[] envProps = aoDesc.getConfigProperties().toArray();

        //Add default config properties to aor
        //Override them if same config properties are provided by the user
        for (int i = 0; i < envProps.length; i++) {
            EnvironmentProperty envProp = (EnvironmentProperty) envProps[i];
            String name = envProp.getName();
            String userValue = (String) props.remove(name);
            if (userValue != null)
                aor.addConfigProperty(new EnvironmentProperty(
                        name, userValue, userValue, envProp.getType()));
            else
                aor.addConfigProperty(envProp);
        }

        //Add non-default config properties provided by the user to aor
        Iterator iter = props.keySet().iterator();
        while (iter.hasNext()) {
            String name = (String) iter.next();
            String userValue = props.getProperty(name);
            if (userValue != null)
                aor.addConfigProperty(new EnvironmentProperty(
                        name, userValue, userValue));

        }

        // bind to JNDI namespace
        try {

            Reference ref = aor.createAdminObjectReference();
            GlassfishNamingManager nm = ConnectorRuntime.getRuntime().getNamingManager();
            nm.publishObject(jndiName, ref, true);

        } catch (NamingException ex) {
            String i18nMsg = localStrings.getString(
                    "aira.cannot_bind_admin_obj");
            throw new ConnectorRuntimeException(i18nMsg);
        }
    }

}
