/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2010 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 *
 * Contributor(s):
 *
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package org.glassfish.ejb.embedded;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Set;
import java.util.logging.Logger;
import java.util.logging.Level;

import javax.ejb.EJBException;

import org.glassfish.api.embedded.ScatteredArchive;
import org.glassfish.deployment.common.ModuleExploder;
import com.sun.enterprise.util.io.FileUtils;
import com.sun.logging.LogDomains;
import com.sun.ejb.containers.EjbContainerUtilImpl;

/**
 * Wrapper that allows to distinguish between an EJB module and a library reference.
 *
 * @author Marina Vatkina
 */
public class DeploymentElement {

    // Use Bundle from another package
    private static final Logger _logger =
            LogDomains.getLogger(EjbContainerUtilImpl.class, LogDomains.EJB_LOGGER);

    private File element;
    private boolean isEJBModule;

    DeploymentElement (File element, boolean isEJBModule) {
        this.element  = element;
        this.isEJBModule  = isEJBModule;
    }

    File getElement() {
        return element;
    }

    boolean isEJBModule() {
        return isEJBModule;
    }

    public static boolean hasEJBModule(Set<DeploymentElement> modules) {
        for (DeploymentElement module : modules) {
            if (module.isEJBModule()) {
                return true;
            }
        }
        return false;
    }

    public static boolean hasLibrary(Set<DeploymentElement> modules) {
        for (DeploymentElement module : modules) {
            if (!module.isEJBModule()) {
                return true;
            }
        }
        return false;
    }

    public static int countEJBModules(Set<DeploymentElement> modules) {
        int result = 0;
        for (DeploymentElement module : modules) {
            if (module.isEJBModule()) {
                ++result;
            }
        }
        return result;
    }


    /**
     * Create deployable application from a Set of DeploymentElements.
     * @param modules the Set of DeploymentElements.
     * @return deployable application.
     */
    public static ResultApplication getOrCreateApplication(Set<DeploymentElement> modules)
            throws EJBException, IOException {
        Object result = null;
        boolean deleteOnExit = false;
        if (modules == null || modules.size() == 0 || !DeploymentElement.hasEJBModule(modules)) {
            _logger.info("[DeploymentElement] No modules found");
        } else if (modules.size() == 1) {
            // Single EJB module
            result = modules.iterator().next().getElement();
        } else if (DeploymentElement.countEJBModules(modules) == 1) {
            // EJB molule with libraries - create ScatteredArchive
            String aName = null;
            Collection<URL> archives = new ArrayList<URL>();
            for (DeploymentElement m : modules) {
                boolean isEJBModule = m.isEJBModule();
                File f = m.getElement();
                String name = f.getName();
                if (_logger.isLoggable(Level.INFO)) {
                    _logger.info("[DeploymentElement] adding " + ((isEJBModule)? "EJB module" : "library") + " to ScatteredArchive " + name);
                }
    
                if (isEJBModule) {
                    // Need to give archive some meaningful name
                    aName = name;
                }
                archives.add(f.toURI().toURL());
            }
            ScatteredArchive.Builder saBuilder = new ScatteredArchive.Builder(aName,
                    Collections.unmodifiableCollection(archives));
            result = saBuilder.buildJar();
        } else {
            // Create a temp dir by creating a temp file first, then
            // delete the file and create a directory in its place.
            File resultFile = File.createTempFile("ejb-app", "");
            File lib = null;
            if (resultFile.delete() && resultFile.mkdirs()) {
                if (_logger.isLoggable(Level.FINE)) {
                    _logger.fine("[DeploymentElement] temp dir created at " + resultFile.getAbsolutePath());
                }

                // Create lib dir if there are library entries
                if (DeploymentElement.hasLibrary(modules)) {
                    if (_logger.isLoggable(Level.FINE)) {
                        _logger.fine("[DeploymentElement] lib dir added ... ");
                    }
                    lib = new File(resultFile, "lib");
                }

            } else {
                throw new EJBException("Not able to create temp dir " + resultFile.getAbsolutePath ());
            }

            if (System.getProperty(EJBContainerProviderImpl.KEEP_TEMPORARY_FILES) == null) {
                 deleteOnExit = true;
            }

            // Copy module directories and explode module jars
            int duplicate_dir_counter = 0;
            for (DeploymentElement m : modules) {
                File f = m.getElement();
                String filename = f.toURI().getSchemeSpecificPart();
                if (filename.endsWith(File.separator) || filename.endsWith("/")) {
                    int length = filename.length();
                    filename = filename.substring(0, length - 1);
                }

                int lastpart = filename.lastIndexOf(File.separatorChar);
                if (lastpart == -1) {
                    lastpart = filename.lastIndexOf('/');
                }
                String name = filename.substring(lastpart + 1);
                if (_logger.isLoggable(Level.FINE)) {
                    _logger.fine("[DeploymentElement] Converted file name: " + filename + " to " + name);
                }

                File base = (m.isEJBModule())? resultFile : lib;
                if (!f.isDirectory() && m.isEJBModule()) { 
                    File out = new File(base, FileUtils.makeFriendlyFilename(name));
                    if (_logger.isLoggable(Level.FINE)) {
                        _logger.fine("[DeploymentElement] Exploding jar to: " + out);
                    }
                    ModuleExploder.explodeJar(f, out);
                } else {
                    if (f.isDirectory()) { 
                        name = name + (m.isEJBModule()? "_jar" : ".jar");
                    }
                    File out = new File(base, name);
                    if (out.exists()) {
                        out = new File(base, "d__" + ++duplicate_dir_counter + "__" + name);
                    }
                    if (_logger.isLoggable(Level.FINE)) {
                        _logger.fine("[DeploymentElement] Copying element to: " + out);
                    }
                    FileUtils.copy(f, out);
                }

                result = resultFile;
            }
        }
        return new ResultApplication(result, deleteOnExit);
    }

    protected static class ResultApplication {
        private boolean deleteOnExit = false;
        private Object app = null;

        ResultApplication (Object app) {
            this.app = app;
        }

        ResultApplication (Object app, boolean deleteOnExit) {
            this.app = app;
            this.deleteOnExit = deleteOnExit;
        }

        Object getApplication() {
            return app;
        }

        boolean deleteOnExit() {
            return deleteOnExit;
        }
    }

}
