/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2013 Oracle and/or its affiliates. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License.  You can
 * obtain a copy of the License at
 * http://glassfish.java.net/public/CDDL+GPL_1_1.html
 * or packager/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at packager/legal/LICENSE.txt.
 *
 * GPL Classpath Exception:
 * Oracle designates this particular file as subject to the "Classpath"
 * exception as provided by Oracle in the GPL Version 2 section of the License
 * file that accompanied this code.
 *
 * Modifications:
 * If applicable, add the following below the License Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyright [year] [name of copyright owner]"
 *
 * Contributor(s):
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.glassfish.jersey.client;

import java.util.Collections;

import javax.ws.rs.ext.ReaderInterceptor;
import javax.ws.rs.ext.WriterInterceptor;

import javax.inject.Inject;
import javax.inject.Provider;

import org.glassfish.jersey.internal.inject.Providers;
import org.glassfish.jersey.internal.util.collection.Ref;
import org.glassfish.jersey.message.MessageBodyWorkers;
import org.glassfish.jersey.model.internal.RankedComparator;

import org.glassfish.hk2.api.ServiceLocator;

import com.google.common.base.Function;
import com.google.common.collect.Lists;

/**
 * Function that can be put to an acceptor chain to properly initialize
 * the client-side request-scoped processing injection for the current
 * request and response exchange.
 *
 * @author Marek Potociar (marek.potociar at oracle.com)
 */
public class RequestProcessingInitializationStage implements Function<ClientRequest, ClientRequest> {
    private final Provider<Ref<ClientRequest>> requestRefProvider;
    private final Provider<MessageBodyWorkers> workersProvider;
    private final Iterable<WriterInterceptor> writerInterceptors;
    private final Iterable<ReaderInterceptor> readerInterceptors;

    /**
     * Create new {@link org.glassfish.jersey.message.MessageBodyWorkers} initialization function
     * for requests and responses.
     *
     * @param requestRefProvider client request context reference injection provider.
     * @param workersProvider message body workers injection provider.
     * @param locator service locator.
     */
    @Inject
    public RequestProcessingInitializationStage(
            Provider<Ref<ClientRequest>> requestRefProvider,
            Provider<MessageBodyWorkers> workersProvider,
            ServiceLocator locator) {
        this.requestRefProvider = requestRefProvider;
        this.workersProvider = workersProvider;
        writerInterceptors = Collections.unmodifiableList(Lists.newArrayList(Providers.getAllProviders(locator,
                WriterInterceptor.class, new RankedComparator<WriterInterceptor>())));
        readerInterceptors = Collections.unmodifiableList(Lists.newArrayList(Providers.getAllProviders(locator,
                ReaderInterceptor.class, new RankedComparator<ReaderInterceptor>())));
    }

    @Override
    public ClientRequest apply(ClientRequest requestContext) {
        requestRefProvider.get().set(requestContext);
        requestContext.setWorkers(workersProvider.get());
        requestContext.setWriterInterceptors(writerInterceptors);
        requestContext.setReaderInterceptors(readerInterceptors);

        return requestContext;
    }
}
