/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012 Oracle and/or its affiliates. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License.  You can
 * obtain a copy of the License at
 * http://glassfish.java.net/public/CDDL+GPL_1_1.html
 * or packager/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at packager/legal/LICENSE.txt.
 *
 * GPL Classpath Exception:
 * Oracle designates this particular file as subject to the "Classpath"
 * exception as provided by Oracle in the GPL Version 2 section of the License
 * file that accompanied this code.
 *
 * Modifications:
 * If applicable, add the following below the License Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyright [year] [name of copyright owner]"
 *
 * Contributor(s):
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.glassfish.jersey.message.internal;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.annotation.Annotation;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.ext.MessageBodyWriter;
import javax.ws.rs.ext.WriterInterceptor;
import javax.ws.rs.ext.WriterInterceptorContext;

import org.glassfish.jersey.internal.LocalizationMessages;
import org.glassfish.jersey.internal.ProcessingException;
import org.glassfish.jersey.internal.PropertiesDelegate;
import org.glassfish.jersey.message.MessageBodyWorkers;
import org.glassfish.jersey.message.MessageBodyWorkers.MessageBodySizeCallback;

/**
 * Represents writer interceptor chain executor for both client and server side.
 * It constructs wrapped interceptor chain and invokes it. At the end of the chain
 * a {@link MessageBodyWriter message body writer} execution interceptor is inserted,
 * which writes entity to the output stream provided by the chain.
 *
 * @author Miroslav Fuksa (miroslav.fuksa at oracle.com)
 * @author Jakub Podlesak (jakub.podlesak at oracle.com)
 */
public final class WriterInterceptorExecutor extends InterceptorExecutor implements WriterInterceptorContext {

    /**
     * Defines property, which is used to pass a list of writer interceptors
     * to the executor via {@link PropertiesDelegate}.
     */
    public static final String INTERCEPTORS = "jersey.runtime.writer.interceptors";

    private OutputStream outputStream;
    private final MultivaluedMap<String, Object> headers;
    private Object entity;

    private final Iterator<WriterInterceptor> iterator;

    /**
     * Constructs a new executor to write given type to provided {@link InputStream entityStream}.
     * List of interceptors to be used is taken from given {@link MessageBodyWorkers workers} instance
     * unless {@value #INTERCEPTORS} property is set in {@link PropertiesDelegate propertiesDelegate}.
     * If such a property is present, the executor tries to cast it to <code>List&lt;WriterInterceptor&gt;</code>
     * and the list is then used to build the interceptor chain.
     *
     * @param entity entity object to be processed.
     * @param rawType     raw Java entity type.
     * @param type        generic Java entity type.
     * @param annotations array of annotations on the declaration of the artifact
     *            that will be initialized with the produced instance. E.g. if the message
     *            body is to be converted into a method parameter, this will be the
     *            annotations on that parameter returned by
     *            {@code Method.getParameterAnnotations}.
     * @param mediaType media type of the HTTP entity.
     * @param headers mutable HTTP headers associated with HTTP entity.
     * @param propertiesDelegate request-scoped properties delegate.
     * @param entityStream {@link java.io.InputStream} from which an entity will be read. The stream is not
     *            closed after reading the entity.
     * @param workers {@link MessageBodyWorkers Message body workers}.
     * @param sizeCallback {@link MessageBodySizeCallback} instance. Can be null.
     * @param intercept if true, user interceptors will be executed. Otherwise only
     *            {@link ExceptionWrapperInterceptor exception wrapping interceptor} will
     *            be executed on the client side.
     * @param writeEntity if true, the entity will be written. Otherwise only headers will
     *            be written to the underlying {@link OutputStream}.
     */
    public WriterInterceptorExecutor(Object entity, Class<?> rawType, Type type, Annotation[] annotations, MediaType mediaType,
            MultivaluedMap<String, Object> headers, PropertiesDelegate propertiesDelegate, OutputStream entityStream,
            MessageBodyWorkers workers, MessageBodySizeCallback sizeCallback, boolean intercept, boolean writeEntity) {

        super(rawType, type, annotations, mediaType, propertiesDelegate);
        this.entity = entity;
        this.headers = headers;
        this.outputStream = entityStream;

        final List<WriterInterceptor> effectiveInterceptors = new ArrayList<WriterInterceptor>();

        final Object writerInterceptorsProperty = propertiesDelegate.getProperty(INTERCEPTORS);
        final Collection<WriterInterceptor> writerInterceptors =
                (writerInterceptorsProperty != null) ? (Collection<WriterInterceptor>)writerInterceptorsProperty : workers.getWriterInterceptors();

        for (WriterInterceptor interceptor : writerInterceptors) {
            if (intercept || (interceptor instanceof ExceptionWrapperInterceptor)) {
                effectiveInterceptors.add(interceptor);
            }
        }
        effectiveInterceptors.add(new TerminalWriterInterceptor(workers, sizeCallback, writeEntity));

        this.iterator = effectiveInterceptors.iterator();
    }

    /**
     * Returns next {@link WriterInterceptor interceptor} in the chain. Stateful method.
     *
     * @return Next interceptor.
     */
    public WriterInterceptor getNextInterceptor() {
        if (!iterator.hasNext()) {
            return null;
        }
        return iterator.next();
    }

    /**
     * Starts the interceptor chain execution.
     */
    @Override
    @SuppressWarnings("unchecked")
    public void proceed() throws IOException {
        WriterInterceptor nextInterceptor = getNextInterceptor();
        if (nextInterceptor == null) {
            throw new ProcessingException(LocalizationMessages.ERROR_INTERCEPTOR_WRITER_PROCEED());
        }
        nextInterceptor.aroundWriteTo(this);
    }

    @Override
    public Object getEntity() {
        return entity;
    }

    @Override
    public void setEntity(Object entity) {
        this.entity = entity;
    }

    @Override
    public OutputStream getOutputStream() {
        return this.outputStream;
    }

    @Override
    public void setOutputStream(OutputStream os) {
        this.outputStream = os;

    }

    @Override
    public MultivaluedMap<String, Object> getHeaders() {
        return this.headers;
    }

    /**
     * Terminal writer interceptor which choose the appropriate {@link MessageBodyWriter}
     * and writes the entity to the output stream. The order of actions is the following: <br>
     * 1. choose the appropriate {@link MessageBodyWriter} <br>
     * 2. if callback is defined then it retrieves size and passes it to the callback <br>
     * 3. writes the entity to the output stream <br>
     *
     */
    private static class TerminalWriterInterceptor implements WriterInterceptor {
        private final MessageBodyWorkers workers;
        private final MessageBodySizeCallback sizeCallback;
        private final boolean writeEntity;

        public TerminalWriterInterceptor(MessageBodyWorkers workers, MessageBodySizeCallback sizeCallback, boolean writeEntity) {
            super();
            this.workers = workers;
            this.sizeCallback = sizeCallback;
            this.writeEntity = writeEntity;
        }

        @Override
        @SuppressWarnings("unchecked")
        public void aroundWriteTo(WriterInterceptorContext context) throws WebApplicationException, IOException {

            final MessageBodyWriter writer = workers.getMessageBodyWriter(context.getType(), context.getGenericType(),
                    context.getAnnotations(), context.getMediaType());
            if (writer == null) {
                throw new MessageBodyProviderNotFoundException(LocalizationMessages.ERROR_NOTFOUND_MESSAGEBODYWRITER(
                        context.getMediaType(), context.getType(), context.getGenericType()));
            }
            if (sizeCallback != null) {
                long size = writer.getSize(context.getEntity(), context.getType(), context.getGenericType(),
                        context.getAnnotations(), context.getMediaType());
                sizeCallback.onRequestEntitySize(size);
            }

            if(writeEntity) {
                writer.writeTo(context.getEntity(), context.getType(), context.getGenericType(), context.getAnnotations(),
                        context.getMediaType(), context.getHeaders(), context.getOutputStream());
            }
        }
    }

}
