/*
 * Decompiled with CFR 0.152.
 */
package org.glassfish.security.common;

import com.sun.enterprise.util.Utility;
import com.sun.enterprise.util.i18n.StringManager;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import org.glassfish.security.common.SSHA;
import org.glassfish.security.common.SharedSecureRandom;
import org.glassfish.security.common.UserPrincipal;

public final class FileRealmHelper {
    public static final String PARAM_KEYFILE = "file";
    private static final String FIELD_SEP = ";";
    private static final String GROUP_SEP = ",";
    private static final String COMMENT = "#";
    public static final String MISC_VALID_CHARS = "_-.";
    private static final int SALT_SIZE = 8;
    private final HashMap<String, User> userTable = new HashMap();
    private final HashMap<String, Integer> groupSizeMap = new HashMap();
    private final File keyfile;
    private static final String SSHA_TAG = "{SSHA}";
    private static final String algoSHA = "SHA";
    private static final String algoSHA256 = "SHA-256";
    private static final String resetKey = "RESET";
    private static final StringManager sm = StringManager.getManager(FileRealmHelper.class);

    public FileRealmHelper(String keyfileName) throws IOException {
        this.keyfile = new File(keyfileName);
        if (!this.keyfile.exists() && !this.keyfile.createNewFile()) {
            throw new IOException(sm.getString("filerealm.badwrite", keyfileName));
        }
        this.loadKeyFile();
    }

    public Set<String> getUserNames() {
        return this.userTable.keySet();
    }

    public User getUser(String name) {
        return this.userTable.get(name);
    }

    public Set<String> getGroupNames() {
        return this.groupSizeMap.keySet();
    }

    public String[] getGroupNames(String username) {
        User user = this.userTable.get(username);
        if (user == null) {
            return null;
        }
        return user.getGroups();
    }

    public String[] authenticate(String username, char[] password) {
        User user = this.userTable.get(username);
        if (user == null) {
            return null;
        }
        if (resetKey.equals(user.getAlgo())) {
            return null;
        }
        boolean ok = false;
        try {
            ok = SSHA.verify(user.getSalt(), user.getHash(), Utility.convertCharArrayToByteArray(password, StandardCharsets.UTF_8), user.getAlgo());
        }
        catch (Exception e) {
            return null;
        }
        if (!ok) {
            return null;
        }
        return user.getGroups();
    }

    public boolean hasAuthenticatableUser() {
        for (User user : this.userTable.values()) {
            if (resetKey.equals(user.getAlgo())) continue;
            return true;
        }
        return false;
    }

    private static boolean isValid(String s, boolean userName) {
        for (int i = 0; i < s.length(); ++i) {
            char c = s.charAt(i);
            if (Character.isLetterOrDigit(c) || Character.isWhitespace(c) || MISC_VALID_CHARS.indexOf(c) != -1 || userName && c == '@') continue;
            return false;
        }
        return true;
    }

    public static void validateUserName(String name) throws IllegalArgumentException {
        if (name == null || name.length() == 0) {
            String msg = sm.getString("filerealm.noname");
            throw new IllegalArgumentException(msg);
        }
        if (!FileRealmHelper.isValid(name, true)) {
            String msg = sm.getString("filerealm.badname", name);
            throw new IllegalArgumentException(msg);
        }
        if (!name.equals(name.trim())) {
            String msg = sm.getString("filerealm.badspaces", name);
            throw new IllegalArgumentException(msg);
        }
    }

    public static void validatePassword(char[] pwd) throws IllegalArgumentException {
        if (Arrays.equals(null, pwd)) {
            String msg = sm.getString("filerealm.emptypwd");
            throw new IllegalArgumentException(msg);
        }
        for (char c : pwd) {
            if (!Character.isSpaceChar(c)) continue;
            String msg = sm.getString("filerealm.badspacespwd");
            throw new IllegalArgumentException(msg);
        }
    }

    public static void validateGroupName(String group) throws IllegalArgumentException {
        if (group == null || group.length() == 0) {
            String msg = sm.getString("filerealm.nogroup");
            throw new IllegalArgumentException(msg);
        }
        if (!FileRealmHelper.isValid(group, false)) {
            String msg = sm.getString("filerealm.badchars", group);
            throw new IllegalArgumentException(msg);
        }
        if (!group.equals(group.trim())) {
            String msg = sm.getString("filerealm.badspaces", group);
            throw new IllegalArgumentException(msg);
        }
    }

    public static void validateGroupList(String[] groupList) throws IllegalArgumentException {
        if (groupList == null || groupList.length == 0) {
            return;
        }
        for (String element : groupList) {
            FileRealmHelper.validateGroupName(element);
        }
    }

    public synchronized void addUser(String name, char[] password, String[] groupList) throws IllegalArgumentException {
        FileRealmHelper.validateUserName(name);
        FileRealmHelper.validatePassword(password);
        FileRealmHelper.validateGroupList(groupList);
        if (this.userTable.containsKey(name)) {
            String msg = sm.getString("filerealm.dupuser", name);
            throw new IllegalArgumentException(msg);
        }
        this.addGroupNames(groupList);
        User ud = FileRealmHelper.createNewUser(name, password, groupList);
        this.userTable.put(name, ud);
    }

    public synchronized void removeUser(String name) throws IllegalArgumentException {
        if (!this.userTable.containsKey(name)) {
            String msg = sm.getString("filerealm.nouser", name);
            throw new IllegalArgumentException(msg);
        }
        User oldUser = this.userTable.get(name);
        this.userTable.remove(name);
        this.reduceGroups(oldUser.getGroups());
    }

    public synchronized void updateUser(String name, String newName, char[] password, String[] groups) throws IllegalArgumentException {
        FileRealmHelper.validateUserName(name);
        if (!this.userTable.containsKey(name)) {
            String msg = sm.getString("filerealm.nouser", name);
            throw new IllegalArgumentException(msg);
        }
        FileRealmHelper.validateUserName(newName);
        FileRealmHelper.validateGroupList(groups);
        if (password != null) {
            FileRealmHelper.validatePassword(password);
        }
        if (!name.equals(newName) && this.userTable.containsKey(newName)) {
            String msg = sm.getString("filerealm.dupuser", name);
            throw new IllegalArgumentException(msg);
        }
        User oldUser = this.userTable.get(name);
        assert (oldUser != null);
        User newUser = new User(newName);
        if (groups != null) {
            this.changeGroups(oldUser.getGroups(), groups);
            newUser.setGroups(groups);
        } else {
            newUser.setGroups(oldUser.getGroups());
        }
        if (password == null) {
            newUser.setSalt(oldUser.getSalt());
            newUser.setHash(oldUser.getHash());
            if (oldUser.getAlgo().equals(resetKey)) {
                newUser.setAlgo(algoSHA256);
            } else {
                newUser.setAlgo(oldUser.getAlgo());
            }
        } else {
            FileRealmHelper.setPassword(newUser, password);
            newUser.setAlgo(algoSHA256);
        }
        this.userTable.remove(name);
        this.userTable.put(newName, newUser);
    }

    public synchronized void persist() throws IOException {
        try (FileOutputStream out = new FileOutputStream(this.keyfile);){
            for (Map.Entry<String, User> uval : this.userTable.entrySet()) {
                String algo = uval.getValue().getAlgo();
                String entry = FileRealmHelper.encodeUser(uval.getKey(), uval.getValue(), algo);
                out.write(entry.getBytes(StandardCharsets.UTF_8));
            }
        }
        catch (IOException e) {
            throw e;
        }
        catch (Exception e) {
            String msg = sm.getString("filerealm.badwrite", this.keyfile);
            throw new IOException(msg, e);
        }
    }

    private void addGroupNames(String[] groupList) {
        if (groupList != null) {
            String[] stringArray = groupList;
            int n = stringArray.length;
            for (int i = 0; i < n; ++i) {
                String element;
                Integer groupSize = this.groupSizeMap.get(element = stringArray[i]);
                this.groupSizeMap.put(element, groupSize != null ? groupSize + 1 : 1);
            }
        }
    }

    private void reduceGroups(String[] groupList) {
        if (groupList != null) {
            for (String element : groupList) {
                Integer groupSize = this.groupSizeMap.get(element);
                if (groupSize == null) continue;
                int gpSize = groupSize - 1;
                if (gpSize > 0) {
                    this.groupSizeMap.put(element, gpSize);
                    continue;
                }
                this.groupSizeMap.remove(element);
            }
        }
    }

    private void changeGroups(String[] oldGroupList, String[] newGroupList) {
        this.addGroupNames(newGroupList);
        this.reduceGroups(oldGroupList);
    }

    private void loadKeyFile() throws IOException {
        try (BufferedReader input = new BufferedReader(new FileReader(this.keyfile, StandardCharsets.UTF_8));){
            while (input.ready()) {
                String line = input.readLine();
                if (line == null || line.startsWith(COMMENT) || line.indexOf(FIELD_SEP) < 0) continue;
                User ud = FileRealmHelper.decodeUser(line, this.groupSizeMap);
                this.userTable.put(ud.getName(), ud);
            }
        }
        catch (Exception e) {
            throw new IOException("Could not load key file " + this.keyfile, e);
        }
    }

    private static String encodeUser(String name, User ud, String algo) {
        StringBuilder sb = new StringBuilder();
        sb.append(name);
        sb.append(FIELD_SEP);
        if (resetKey.equals(algo)) {
            sb.append(resetKey);
        } else {
            String ssha = SSHA.encode(ud.getSalt(), ud.getHash(), algo);
            sb.append(ssha);
        }
        sb.append(FIELD_SEP);
        String[] groups = ud.getGroups();
        if (groups != null) {
            for (int grp = 0; grp < groups.length; ++grp) {
                if (grp > 0) {
                    sb.append(GROUP_SEP);
                }
                sb.append(groups[grp]);
            }
        }
        sb.append("\n");
        return sb.toString();
    }

    private static User decodeUser(String encodedLine, Map newGroupSizeMap) throws IllegalArgumentException {
        StringTokenizer st = new StringTokenizer(encodedLine, FIELD_SEP);
        String algo = algoSHA256;
        String user = null;
        String pwdInfo = null;
        String groupList = null;
        try {
            user = st.nextToken();
            pwdInfo = st.nextToken();
        }
        catch (Exception e) {
            String msg = sm.getString("filerealm.syntaxerror", encodedLine);
            throw new IllegalArgumentException(msg);
        }
        if (st.hasMoreTokens()) {
            groupList = st.nextToken();
        }
        User ud = new User(user);
        if (resetKey.equals(pwdInfo)) {
            ud.setAlgo(resetKey);
        } else {
            if (encodedLine.contains(SSHA_TAG)) {
                algo = algoSHA;
            }
            int resultLength = 32;
            if (algoSHA.equals(algo)) {
                resultLength = 20;
            }
            byte[] hash = new byte[resultLength];
            byte[] salt = SSHA.decode(pwdInfo, hash, algo);
            ud.setHash(hash);
            ud.setSalt(salt);
            ud.setAlgo(algo);
        }
        ArrayList<String> membership = new ArrayList<String>();
        if (groupList != null) {
            StringTokenizer gst = new StringTokenizer(groupList, GROUP_SEP);
            while (gst.hasMoreTokens()) {
                String g = gst.nextToken();
                membership.add(g);
                Integer groupSize = (Integer)newGroupSizeMap.get(g);
                newGroupSizeMap.put(g, groupSize != null ? groupSize + 1 : 1);
            }
        }
        ud.setGroups(membership.toArray(new String[membership.size()]));
        return ud;
    }

    private static User createNewUser(String name, char[] pwd, String[] groups) {
        User ud = new User(name);
        if (groups == null) {
            groups = new String[]{};
        }
        ud.setGroups(groups);
        ud.setAlgo(algoSHA256);
        FileRealmHelper.setPassword(ud, pwd);
        return ud;
    }

    private static void setPassword(User user, char[] pwd) throws IllegalArgumentException {
        byte[] pwdBytes;
        assert (user != null);
        try {
            pwdBytes = Utility.convertCharArrayToByteArray(pwd, StandardCharsets.UTF_8);
        }
        catch (Exception ex) {
            throw new IllegalArgumentException(ex);
        }
        byte[] salt = new byte[8];
        SharedSecureRandom.SECURE_RANDOM.nextBytes(salt);
        user.setSalt(salt);
        String algo = user.getAlgo();
        if (algo == null) {
            algo = algoSHA256;
        }
        byte[] hash = SSHA.compute(salt, pwdBytes, algo);
        user.setHash(hash);
    }

    public static class User
    implements UserPrincipal {
        private static final long serialVersionUID = 1L;
        private final String name;
        private String[] groups;
        private String realm;
        private byte[] salt;
        private byte[] hash;
        private String algo;

        public User(String name) {
            this.name = name;
        }

        public User(String name, String[] groups, String realm, byte[] salt, byte[] hash, String algo) {
            this.name = name;
            this.groups = groups;
            this.realm = realm;
            this.hash = hash;
            this.salt = salt;
            this.algo = algo;
        }

        @Override
        public String getName() {
            return this.name;
        }

        public byte[] getSalt() {
            return this.salt;
        }

        public void setSalt(byte[] salt) {
            this.salt = salt;
        }

        public byte[] getHash() {
            return this.hash;
        }

        public void setHash(byte[] hash) {
            this.hash = hash;
        }

        public String[] getGroups() {
            return this.groups;
        }

        public void setGroups(String[] grp) {
            this.groups = grp;
        }

        public String getAlgo() {
            return this.algo;
        }

        public void setAlgo(String algo) {
            this.algo = algo;
        }

        @Override
        public boolean equals(Object obj) {
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            User other = (User)obj;
            if (!Arrays.deepEquals(this.groups, other.groups)) {
                return false;
            }
            if (this.realm == null ? other.realm != null : !this.realm.equals(other.realm)) {
                return false;
            }
            if (!Arrays.equals(this.salt, other.salt)) {
                return false;
            }
            if (!Arrays.equals(this.hash, other.hash)) {
                return false;
            }
            if (this.algo == null ? other.algo != null : !this.algo.equals(other.algo)) {
                return false;
            }
            return super.equals(obj);
        }

        @Override
        public int hashCode() {
            int hc = 5;
            hc = 17 * hc + Arrays.deepHashCode(this.groups);
            hc = 17 * hc + (this.realm == null ? 0 : this.realm.hashCode());
            hc = 17 * hc + Arrays.hashCode(this.salt);
            hc = 17 * hc + Arrays.hashCode(this.hash);
            hc = 17 * hc + (this.algo == null ? 0 : this.algo.hashCode());
            hc = 17 * super.hashCode();
            return hc;
        }

        @Override
        public String toString() {
            return this.getName();
        }
    }
}

