/*
 * Copyright (c) 1997, 2018 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0, which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the
 * Eclipse Public License v. 2.0 are satisfied: GNU General Public License,
 * version 2 with the GNU Classpath Exception, which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 */

package org.glassfish.deployapi;

import org.glassfish.logging.annotation.LogMessageInfo;

import javax.enterprise.deploy.shared.factories.DeploymentFactoryManager;
import javax.enterprise.deploy.spi.factories.DeploymentFactory;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.net.URLClassLoader;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;

/**
 * This singleton object is responsible to resolving all the 
 * DeploymentManagerFactory installed in the RI and register
 * them to the DeploymentManagerFactory
 *
 * @author Jerome Dochez
 */
public class DeploymentFactoryInstaller {
    
    private static DeploymentFactoryInstaller dfInstaller = null;
    
    private final String J2EE_DEPLOYMENT_MANAGER_REPOSITORY = "lib" + File.separator + "deployment";
    private static final String J2EE_DEPLOYMENT_MANAGER = "J2EE-DeploymentFactory-Implementation-Class";
    private static final String J2EE_HOME = "com.sun.enterprise.home";
    
    public static final Logger deplLogger = org.glassfish.deployment.client.AbstractDeploymentFacility.deplLogger;

    @LogMessageInfo(message = "Deployment manager load failure.  Unable to find {0}",cause="A deployment manager is not available.",action="Correct the reference to the deployment manager.", level="SEVERE")
    private static final String NO_DEPLOYMENT_MANAGER = "AS-DEPLOYMENT-04018";

    /** Creates a single instance of DeploymentManagerFactoryResolver */
    private DeploymentFactoryInstaller() {
    }
    
    public static DeploymentFactoryInstaller getInstaller() {
    
        if (dfInstaller==null) {
            DeploymentFactoryInstaller tmpInstaller = new DeploymentFactoryInstaller();
            tmpInstaller.initialize();
            dfInstaller = tmpInstaller; // The use of tmpInstaller insures that dfInstaller is not available to other threads until it is initialized
        }
        return dfInstaller;
    }
    
    /**
     * @return a list of installed deployment manager 
     * implementation archives
     */
    public File[] getListOfDeploymentFactoryFiles() {
        
        File repository = new File(System.getProperty("com.sun.aas.installRoot")+File.separator+
            J2EE_DEPLOYMENT_MANAGER_REPOSITORY);
        
        if (deplLogger.isLoggable(Level.FINE)) {
            deplLogger.fine("J2EE Deployment factory repository = " 
                            + repository.getAbsolutePath());
        }
        if (!repository.exists()) {
            deplLogger.log(Level.SEVERE,
                           NO_DEPLOYMENT_MANAGER,
                           repository.getAbsolutePath());
            return null;
        }
        
        return repository.listFiles();        
    }
    
    protected void installDeploymentFactory(final File installedDM) throws IOException {
        
        if (deplLogger.isLoggable(Level.FINE)) {
            deplLogger.fine("Installing Deployment factory = " 
                            + installedDM.getAbsolutePath());
        }
        
        // let's check first that we indeed have a valid 
        // deployment manager implementation
        
        /*
         *Declare the JarFile and Manifest but populate them inside the first try block.  This way the 
         *jar file can be closed right away to conserve resources.
         */
        Manifest m = null;
        JarFile jarFile = new JarFile(installedDM);
        try {
            m = jarFile.getManifest();
        } finally {
            jarFile.close();
        }
        String className = m.getMainAttributes().getValue(J2EE_DEPLOYMENT_MANAGER);
        final URL[] urls = new URL[]{installedDM.toURI().toURL()};
        URLClassLoader urlClassLoader;
        urlClassLoader = AccessController.doPrivileged(new PrivilegedAction<URLClassLoader>() {
          public URLClassLoader run() {
              return new java.net.URLClassLoader(urls, getClass().getClassLoader());
          }
        });

        Class factory = null;
        try {
            factory=urlClassLoader.loadClass(className);
        } catch (ClassNotFoundException cnfe) {
            deplLogger.log(Level.SEVERE,
                           NO_DEPLOYMENT_MANAGER,
                           className);
            throw new IllegalArgumentException(className + " is not present in the " + installedDM.getName());
        }
        
        // Ok we have the class, let's instanciate it, check it and 
        // if everything is fine, register it to the DeploymentFactoryManager
        Object df = null;
        try {            
            df = factory.newInstance();
        } catch (Exception ie) {
            LogRecord lr = new LogRecord(Level.SEVERE, NO_DEPLOYMENT_MANAGER);
            Object args[] = { className };
            lr.setParameters(args);
            lr.setThrown(ie);
            deplLogger.log(lr);
            throw new IllegalArgumentException("Cannot install " + installedDM.getName());
        }
        if (df instanceof DeploymentFactory) {
            DeploymentFactoryManager.getInstance().registerDeploymentFactory((DeploymentFactory) df);
        } else {
            throw new IllegalArgumentException("The " + className + 
                " declared as a DeploymentFactory does implement the DeploymentFactory interface");
        }
    }
    
    protected void initialize() {
                
        File[] elligibleFiles = getListOfDeploymentFactoryFiles();
        if (elligibleFiles==null) {
            return;
        }
           
        for (int i=0;i<elligibleFiles.length;i++) {
            try {
                installDeploymentFactory(elligibleFiles[i]);
            } catch(Exception ioe) {
              LogRecord lr = new LogRecord(Level.SEVERE, NO_DEPLOYMENT_MANAGER);
              Object args[] = { elligibleFiles[i].getName() };
              lr.setParameters(args);
              lr.setThrown(ioe);
              deplLogger.log(lr);
            }
        }        
    }       
}
