/*
 * Copyright (c) 2009, 2018 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0, which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the
 * Eclipse Public License v. 2.0 are satisfied: GNU General Public License,
 * version 2 with the GNU Classpath Exception, which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 */

package org.glassfish.webservices.deployment;

import com.sun.enterprise.deployment.Application;
import com.sun.enterprise.deployment.WebServiceEndpoint;
import com.sun.xml.ws.api.server.WSEndpoint;
import com.sun.xml.ws.transport.http.servlet.ServletAdapter;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import javax.xml.namespace.QName;

import org.glassfish.external.statistics.impl.StatisticImpl;
import org.glassfish.gmbal.Description;
import org.glassfish.gmbal.ManagedAttribute;
import org.glassfish.gmbal.ManagedData;

/**
 * 109 and sun-jaxws.xml style deployed endpoint's info.
 *
 * @author Jitendra Kotamraju
 */
@ManagedData
@Description("109 deployed endpoint info")
public class DeployedEndpointData extends StatisticImpl implements Map {

    @ManagedAttribute
    @Description("Application Name")
    public final String appName;

    @ManagedAttribute
    @Description("Endpoint Name")
    public final String endpointName;

    @ManagedAttribute
    @Description("Target Namespace of the Web Service")
    public final String namespace;

    @ManagedAttribute
    @Description("Web Service name")
    public final String serviceName;

    @ManagedAttribute
    @Description("Web Service port name")
    public final String portName;

    @ManagedAttribute
    @Description("Service Implementation Class")
    public final String implClass;

    @ManagedAttribute
    @Description("Address for Web Service")
    public final String address;

    @ManagedAttribute
    @Description("WSDL for Web Service")
    public final String wsdl;

    @ManagedAttribute
    @Description("Tester for Web Service")
    public final String tester;

    @ManagedAttribute
    @Description("Implementation Type: EJB or SERVLET")
    public final String implType;

    @ManagedAttribute
    @Description("Deployment Type: 109 or RI")
    public final String deploymentType;

    private Map<String, String> infoMap = new HashMap<String, String>();

    // 109 deployed endpoint
    public DeployedEndpointData(String path, Application app, WebServiceEndpoint endpoint) {
        super(path, "", "");
        this.appName = app.getAppName();
        this.endpointName = endpoint.getEndpointName();
        this.namespace = endpoint.getServiceName().getNamespaceURI();
        this.serviceName = endpoint.getServiceName().getLocalPart();
        QName pName = endpoint.getWsdlPort();
        this.portName = (pName != null) ? pName.getLocalPart() : "";
        this.implClass = endpoint.implementedByEjbComponent()
                ? endpoint.getEjbComponentImpl().getEjbImplClassName()
                : endpoint.getServletImplClass();
        this.address = path;
        this.wsdl = address+"?wsdl";
        this.tester = address+"?Tester";
        this.implType = endpoint.implementedByEjbComponent() ? "EJB" : "SERVLET";
        this.deploymentType = "109";
        fillStatMap();
    }

    // sun-jaxws.xml deployed endpoint
    public DeployedEndpointData(String path, ServletAdapter adapter) {
        super(path, "", "");
        WSEndpoint endpoint = adapter.getEndpoint();

        this.appName = "";
        this.endpointName = adapter.getName();
        this.namespace = endpoint.getServiceName().getNamespaceURI();
        this.serviceName = endpoint.getServiceName().getLocalPart();
        this.portName = endpoint.getPortName().getLocalPart();
        this.implClass = endpoint.getImplementationClass().getName();
        this.address = path;
        this.wsdl = address+"?wsdl";
        this.tester = "";
        this.implType = "SERVLET";
        this.deploymentType = "RI";
        fillStatMap();
    }

    @Override
    public Map<String, String> getStaticAsMap() {
        return infoMap;
    }

    private void fillStatMap() {
        infoMap.put("appName", appName);
        infoMap.put("endpointName", endpointName);
        infoMap.put("namespace", namespace);
        infoMap.put("serviceName", serviceName);
        infoMap.put("portName", portName);
        infoMap.put("implClass", implClass);
        infoMap.put("address", address);
        infoMap.put("wsdl", wsdl);
        infoMap.put("tester", tester);
        infoMap.put("implType", implType);
        infoMap.put("deploymentType", deploymentType);
    }

    @Override
    public int size() {
        return infoMap.size();
    }

    @Override
    public boolean isEmpty() {
        return infoMap.isEmpty();
    }

    @Override
    public boolean containsKey(Object key) {
        return infoMap.containsKey(key);
    }

    @Override
    public boolean containsValue(Object value) {
        return infoMap.containsValue(value);
    }

    @Override
    public Object get(Object key) {
        return infoMap.get(key);
    }

    @Override
    public Object put(Object key, Object value) {
        throw new UnsupportedOperationException("DeployedEndpointData is a read-only map");
    }

    @Override
    public Object remove(Object key) {
        throw new UnsupportedOperationException("DeployedEndpointData is a read-only map");
    }

    @Override
    public void putAll(Map m) {
        throw new UnsupportedOperationException("DeployedEndpointData is a read-only map");
    }

    @Override
    public void clear() {
        throw new UnsupportedOperationException("DeployedEndpointData is a read-only map");
    }

    @Override
    public Set keySet() {
        return infoMap.keySet();
    }

    @Override
    public Collection values() {
        return infoMap.values();
    }

    @Override
    public Set entrySet() {
        return infoMap.entrySet();
    }
}
