/*
 * Copyright (c) 1997, 2018 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0, which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the
 * Eclipse Public License v. 2.0 are satisfied: GNU General Public License,
 * version 2 with the GNU Classpath Exception, which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 */

package com.sun.xml.rpc.processor.generator;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.Iterator;
import java.util.Properties;
import java.util.TreeSet;

import javax.xml.namespace.QName;

import com.sun.xml.rpc.processor.config.Configuration;
import com.sun.xml.rpc.processor.model.Model;
import com.sun.xml.rpc.processor.model.ModelProperties;
import com.sun.xml.rpc.processor.model.Operation;
import com.sun.xml.rpc.processor.model.Port;
import com.sun.xml.rpc.processor.model.Service;
import com.sun.xml.rpc.processor.util.DirectoryUtil;
import com.sun.xml.rpc.processor.util.GeneratedFileInfo;
import com.sun.xml.rpc.processor.util.IndentingWriter;
import com.sun.xml.rpc.processor.util.ProcessorEnvironment;
import com.sun.xml.rpc.soap.SOAPVersion;

/**
 *
 * @author JAX-RPC Development Team
 */
public class ServletConfigGenerator extends GeneratorBase {
    private File configFile;
    private Service currentService;
    private IndentingWriter out;
    private int portCount;
    private TreeSet operations = null;

    public ServletConfigGenerator() {
        configFile = null;
        out = null;
        portCount = 0;
    }

    public GeneratorBase getGenerator(
        Model model,
        Configuration config,
        Properties properties) {
        return new ServletConfigGenerator(model, config, properties);
    }

    public GeneratorBase getGenerator(
        Model model,
        Configuration config,
        Properties properties,
        SOAPVersion ver) {
        return new ServletConfigGenerator(model, config, properties);
    }

    private ServletConfigGenerator(
        Model model,
        Configuration config,
        Properties properties) {
        super(model, config, properties);
        configFile = null;
        out = null;
        portCount = 0;
    }

    protected void preVisitService(Service service) throws Exception {
        try {
            currentService = service;
            String className = service.getName().getLocalPart();
            configFile = configFileForClass(className, nonclassDestDir, env);

            /* Here the filename for the ServletConfig to be geenrated is
               retrieved to be set in the GeneratedFileInfo Object */
            GeneratedFileInfo fi = new GeneratedFileInfo();
            fi.setFile(configFile);
            fi.setType(GeneratorConstants.FILE_TYPE_SERVLET_CONFIG);
            env.addGeneratedFile(fi);

            out =
                new IndentingWriter(
                    new OutputStreamWriter(new FileOutputStream(configFile)));
            portCount = 0;
            out.pln("# This file is generated by wscompile.");
            out.pln();
        } catch (IOException e) {
            fail("cant.write", configFile.toString());
        }
    }

    protected void postVisitService(Service service) throws Exception {
        try {
            out.pln("portcount=" + portCount);
            closeFile();
        } catch (IOException e) {
            fail("cant.write", configFile.toString());
        } finally {
            currentService = null;
        }
    }

    public void visit(Port port) throws Exception {
        int myPortNum = portCount;
        portCount = myPortNum + 1;
        operations = new TreeSet(new StringLenComparator());
        Iterator operations = port.getOperations();
        while (operations.hasNext()) {
            ((Operation) operations.next()).accept(this);
        }
        try {
            String portID = "port" + myPortNum;
            String servant = null;
            servant = port.getJavaInterface().getImpl();
            if (servant == null) {
                servant =
                    "Please specify the servant class for port:"
                        + port.getName().getLocalPart();
            }
            out.pln(portID + ".tie=" + env.getNames().tieFor(port));
            out.pln(portID + ".servant=" + servant);
            out.pln(portID + ".name=" + port.getName().getLocalPart());
            out.pln(
                portID
                    + ".wsdl.targetNamespace="
                    + model.getTargetNamespaceURI());
            out.pln(
                portID
                    + ".wsdl.serviceName="
                    + currentService.getName().getLocalPart());
            QName portName =
                (QName) port.getProperty(
                    ModelProperties.PROPERTY_WSDL_PORT_NAME);
            out.pln(portID + ".wsdl.portName=" + portName.getLocalPart());
        } catch (IOException e) {
            fail("generator.cant.write", configFile.toString());
        }
    }

    protected void visitOperation(Operation operation) throws Exception {
        operations.add(operation);
    }

    private void closeFile() throws IOException {
        if (out != null) {
            out.close();
            out = null;
        }
    }

    private String getBaseName(String s) {
        if (s.endsWith("Port")) {
            return s.substring(0, s.length() - 4);
        } else {
            return s;
        }
    }

    private String getPortName(String s) {
        return getBaseName(s) + "Port";
    }

    /**
     * Return the File object that should be used as the configuration
     * file for the given Java class, using the supplied destination
     * directory for the top of the package hierarchy.
     */
    protected static File configFileForClass(
        String className,
        File destDir,
        ProcessorEnvironment env)
        throws GeneratorException {
        File packageDir =
            DirectoryUtil.getOutputDirectoryFor(className, destDir, env);
        String outputName = Names.stripQualifier(className);

        String outputFileName = outputName + "_Config.properties";
        return new File(packageDir, outputFileName);
    }

    private static class StringLenComparator implements java.util.Comparator {
        public int compare(Object o1, Object o2) {
            int len1, len2;
            len1 = ((Operation) o1).getName().getLocalPart().length();
            len2 = ((Operation) o2).getName().getLocalPart().length();
            return (len1 <= len2) ? -1 : 1;
        }
    }
}
